#include <formatio.h>
#include <userint.h>
#include <stdlib.h>
#include "mxOX2000.h"
#include "mxOX200u.h"

/*= Metrix OX2000 Oscilloscope Application Example Source File ==============*/
/* Original Release: December 1997                                           */
/* By: Olivier Lemaire, Metrix, Annecy-le-Vieux, France                      */
/*     PH. +33(0)450642222  Fax  +33(0)450642299                             */
/*     e-mail: Jan.Sima@vsb.cz                                               */
/*                                                                           */
/* Modification History: None                                                */
/*===========================================================================*/

/*****************************************************************************/
/*= GLOBAL VARIABLE DECLARATIONS ============================================*/
/*****************************************************************************/
static int panelHandle[10];
static unsigned int instrHandle;
static double data[10000];
ViInt16 uirCode;

/*****************************************************************************/
/*= MESSAGE ARRAYS ==========================================================*/
/*****************************************************************************/

/*- Init Panel control helps ------------------------------------------------*/

static char *initCtrlsHelpStr[4] = {"\n\
This control selects type of communication.\n\n\
Valid Range:\n\
VI_OFF (0): Serial mode\n\
VI_ON  (1): GPIB mode (Default Value)\n\n","\n\
This control selects address of the device that is to be\n\
initialized.\n\n\
Valid Range:  Address 1 to 30\n\n\
Default Value :  Address 1","\n\
This control specifies if an ID Query is sent to the\n\
instrument during the initialization procedure.\n\n\
Valid Range:\n\
VI_OFF (0) - Skip Query\n\
VI_ON  (1) - Do Query (Default Value)","\n\
This control specifies if the instrument is to be reset to its\n\
power-on settings during the initialization procedure.\n\n\
Valid Range:\n\
VI_OFF (0) - Don't Reset (Default Value)\n\
VI_ON  (1) - Reset Device"};    

/*- Acquire Waveform Panel control helps ------------------------------------*/     

static char *measCtrlsHelpStr[4] = {"\n\
This control sets the timebase of the oscilloscope.\n\n\
Valid Values:\n\
0 - 2 ns/div\n\
1 - 5 ns/div\n\
2 - 10 ns/div\n\
3 - 20 ns/div\n\
4 - 50 ns/div\n\
5 - 0.1 us/div\n\
6 - 0.2 us/div\n\
7 - 0.5 us/div\n\
8 - 1 us/div\n\
9 - 2 us/div  (Default Value)\n\
10 - 5 us/div\n\
11 - 10 us/div\n\
12 - 20 us/div\n\
13 - 50 us/div\n\
14 - 100 us/div\n\
15 - 200 us/div\n\
16 - 500 us/div\n\
17 - 1 ms/div\n\
18 - 2 ms/div\n\
19 - 5 ms/div\n\
20 - 10 ms/div\n\
21 - 20 ms/div\n\
22 - 50 ms/div\n\
23 - 100 ms/div\n\
24 - 200 ms/div\n\
25 - 500 ms/div\n\
26 - 1 s/div","\n\
This control sets the input range of the selected\n\
channel in units of volts per division.\n\n\
Valid Values:\n\
0 - 2 mV/div\n\
1 - 5 mV/div\n\
2 - 10 mV/div\n\
3 - 20 mV/div\n\
4 - 50 mV/div\n\
5 - 100 mV/div\n\
6 - 200 mV/div\n\
7 - 500 mV/div\n\
8 - 1 V/div\n\
9 - 2 V/div\n\
10 - 5 V/div\n\
11 - 10 V/div","\n\
This control selects the input coupling\n\
of the selected channel.\n\n\
Valid Values:\n\
0 - AC  (Default Value)\n\
1 - DC\n\
2 - GND","\n\
This control displays the waveform acquired\n\
by the instrument. Displayed waveform consists\n\
of samples acquired on channel 1 (trace S1)."};

/*- Background Panel Help ---------------------------------------------------*/

static char backgroundPnlHelpStr[] = {"\n\
This is a simple demo program using the Metrix OX2000\n\
Oscilloscope instrument driver provided in the package."};

/*- Init Panel Help ---------------------------------------------------------*/

static char initPnlHelpStr[] = {"\n\
This panel performs the following initialization actions:\n\n\
- Opens a session to the Default Resource Manager resource and a\n\
session to the specified device using the interface and address\n\
specified in the Resource_Name control.\n\n\
- Performs an identification query on the Instrument.\n\n\
- Resets the instrument to a known state.\n\n\
- Sends initialization commands to the instrument that set any\n\
necessary programmatic variables such as Headers Off, Short\n\
Command form, and Data Transfer Binary to the state necessary\n\
for the operation of the instrument driver.\n\n\
- Returns an Instrument Handle which is used to differentiate\n\
between different sessions of this instrument driver.\n\n\
- Each time this function is invoked a Unique Session is opened.\n\
It is possible to have more than one session open for the same\n\
resource."};

/*- Acquire Waveform Panel Help ---------------------------------------------*/

static char measPnlHelpStr[] = {"\n\
This panel acquires a single channel waveform from the channel 1\n\
of the instrument. Each time you press the button labelled\n\
\"New Measurement\", the instrument's timebase, input voltage\n\
range and input coupling are configured and new waveform is\n\
acquired and displayed on this panel."};

/*= Main Function ===========================================================*/

void main ()
{
    panelHandle[BCKGRND] = LoadPanel (0, "mxOX200u.uir", BCKGRND);
    panelHandle[MEAS] = LoadPanel (panelHandle[BCKGRND], "mxOX200u.uir", MEAS);
    panelHandle[INIT] = LoadPanel (panelHandle[BCKGRND], "mxOX200u.uir", INIT);
    DisplayPanel (panelHandle[BCKGRND]);
    DisplayPanel (panelHandle[INIT]);
    uirCode = RunUserInterface ();
}

/*===========================================================================*/
/* Function: Initialize Instrument                                           */
/* Purpose:  This is a callback function of the Continue button on the       */
/*           Initialize panel. It initializes the instrument and switches to */
/*           the panel Configure.                                            */
/*===========================================================================*/
int initInstrument (int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    short com,addr, rst, id;
    char instrDescr[20];

    switch (event) {
        case EVENT_COMMIT:
            GetCtrlVal (panelHandle[INIT], INIT_COM, &com);
            GetCtrlVal (panelHandle[INIT], INIT_ADDRESS, &addr);
            GetCtrlVal (panelHandle[INIT], INIT_ID, &id);
            GetCtrlVal (panelHandle[INIT], INIT_RST, &rst);
            
            /*- Initializing the instrument ---------------------------------*/
            if(com==1)
                Fmt (instrDescr, "GPIB::%d[b2]", addr);
            else
                Fmt (instrDescr, "ASRL%d[b2]", addr);
            if (mxOX2000_init (instrDescr, id, rst, &instrHandle)) {
                MessagePopup ("Error", "Initialization error. Check your connections and make sure you have the right address.");
                return (0);            
            }
            else {
                HidePanel (panelHandle[INIT]);
                DisplayPanel (panelHandle[MEAS]);
            }
            break;
        case EVENT_RIGHT_CLICK:
            MessagePopup ("Help","This button causes the instrument to be initialized.");
            break;
    }
    return 0;
}

/*===========================================================================*/
/* Function: Control Help                                                    */
/* Purpose:  This is a callback function of all controls that configure the  */
/*           instrument. On the right mouse-click on the control a help      */
/*           window describing its purpose is displayed.                     */
/*===========================================================================*/
int controlHelp (int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    if (event == EVENT_RIGHT_CLICK) {
        if (panel == panelHandle[INIT]) {
            MessagePopup ("Help", initCtrlsHelpStr[control-4]);
        }
        if (panel == panelHandle[MEAS]) {                                
            MessagePopup ("Help", measCtrlsHelpStr[control-4]);
        }
    }
    return 0;
}

/*===========================================================================*/
/* Function: New Measurement                                                 */
/* Purpose:  This is a callback function of the New Measurement button on    */
/*           the Acquire Waveform panel. New waveform is acquired by issuing */
/*           function Example of the Metrix OX2000 Oscilloscope instrument   */
/*           driver.                                                         */
/*===========================================================================*/
int newMeasurement (int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    double dt, x0;
    int error;
    int timebase, vrange, coup, n;
    ViChar errMsg[256], errBuffer[256]; 
    
    switch (event) {
        case EVENT_COMMIT:
            GetCtrlVal (panelHandle[MEAS], MEAS_TIMEBASE, &timebase);
            GetCtrlVal (panelHandle[MEAS], MEAS_VOLTRANGE, &vrange);
            GetCtrlVal (panelHandle[MEAS], MEAS_COUPLING, &coup);
            
            /* Acquiring Waveform -------------------------------------------*/
            if (error = mxOX2000_example (instrHandle, 1, timebase, vrange, coup, data, &n, &x0, &dt)) {
                mxOX2000_errorMessage (VI_NULL, error, errMsg);
                if (error >= 0xBFFF0000) {
                
                    /*- Critical error, no communication --------------------*/
                    Fmt (errMsg, "%s[a]<\nThis program will now terminate!");
                    MessagePopup ("Critical error", errMsg);
                   
                    if ((error = mxOX2000_close (instrHandle)) < 0) {
                        mxOX2000_errorMessage (VI_NULL, error, errMsg);
                        Fmt (errBuffer, "%s<Application failed to close the Instrument Driver.\n\n"
                            "%s\n\nExiting Application.", errMsg);
                        MessagePopup("ERROR!",errBuffer);
                    }
                    else 
                        MessagePopup("Application Aborted","Instrument Driver Closed, Exiting Application.");
                    QuitUserInterface (uirCode); 
                    exit (-1);  
                    break;
                
                }
                else {
                    
                    /*- Non-critical Error, can continue --------------------*/
                    mxOX2000_errorMessage (VI_NULL, error, errMsg);
                    MessagePopup ("Error", errMsg);
                    return 0;                              
                }   
            }
            DeleteGraphPlot (panelHandle[MEAS], MEAS_OUTPUT, -1, VAL_IMMEDIATE_DRAW);
            
            PlotWaveform (panelHandle[MEAS], MEAS_OUTPUT, data, n, VAL_DOUBLE, 1.0, 0.0, 0.0, 1000.0*dt,
                          VAL_THIN_LINE, VAL_EMPTY_SQUARE, VAL_SOLID, 1, VAL_RED);
            break;
        case EVENT_RIGHT_CLICK:
            MessagePopup ("Help","Press this button to configure the instrument and acquire new waveform.");
            break;
    }
    return 0;
}

/*===========================================================================*/
/* Function: Panel Help                                                      */
/* Purpose:  This is a callback function of the menu bar. It displays a help */
/*           window describing panel being used.                             */
/*===========================================================================*/
void panelHelp (int menuBar, int menuItem, void *callbackData,
        int panel)
{
    if (panel == panelHandle[BCKGRND])
        MessagePopup ("Help", backgroundPnlHelpStr);
    if (panel == panelHandle[INIT])
        MessagePopup ("Help", initPnlHelpStr);
    if (panel == panelHandle[MEAS])
        MessagePopup ("Help", measPnlHelpStr);
        
    return;
}

/*===========================================================================*/
/* Function: Launch Close Dialog                                             */
/* Purpose:  This is a callback function of the Quit buttons. It pops-up     */
/*           a dialog panel to choose whether to quit or not.                */
/*===========================================================================*/
int launchClose (int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    ViChar errMsg[256], errBuffer[256]; 
    ViInt32 error = 0;
  
    
    switch (event) {
        case EVENT_COMMIT:
            if (ConfirmPopup ("Close", "Do you really want to quit?")) {
                if (panel != panelHandle[INIT]) {
                    if ((error = mxOX2000_close (instrHandle)) < 0) {
                        mxOX2000_errorMessage (VI_NULL, error, errMsg);
                        Fmt (errBuffer, "%s<Application failed to close the Instrument Driver.\n\n"
                            "%s\n\nExiting Application.", errMsg);
                        MessagePopup("ERROR!",errBuffer);
                    }
                }    
                QuitUserInterface (uirCode);
            }   
            break;
        case EVENT_RIGHT_CLICK:
            MessagePopup ("Help","This button will pop-up a panel to close the instrument.");
            break;
    }
    return 0;
}

/*= End =====================================================================*/
