#include <utility.h>
#include <formatio.h>
#include <visa.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "mxox2000.h"

/*==================================================================================*/
#define OX2000_REVISION    "Rev 1.1, 04/98, CVI 4.0.1"/* Instrument driver revision */
#define BUFFER_SIZE        512L                      /* File I/O buffer size        */
#define INVALID_SAMPLE     0xFFFF                    /* Sample validity criterion   */
#define MAX_MEM_SIZE       200000                    /* Memory needed for file I/O  */
#define INVALID_VALUE      1E100                     /* Invalid sample default value*/
#define TMO_VALUE          5000                      /* Time out value              */
#define HEADER_SIZE        11                        /* Size of binary data header  */
/*==================================================================================*/

/*=OX2000====================================================================*/
/* LabWindows/CVI 4.01 Instrument Driver (VISA I/O)                          */
/* Original Release: April 1998                                              */
/* By: Jrme Manera (Metrix, Annecy, France)                                */
/*                                                                           */                           
/* Metrix Technical Support: Olivier Lemaire                                 */
/*      PH. (33) 04 50 64 22 22     Fax (33) 04 50 64 22 99                  */
/*                                                                           */
/* For LabWindows Technical Support in the United States call:               */
/*              National Instruments, Austin, Texas                          */
/*              Ph (800) 433-3488    Fax (512) 794-5678                      */
/*                                                                           */
/*                                                                           */
/* Modification History: None                                                */
/*                                                                           */
/*===========================================================================*/
 
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS =====================================*/
/*****************************************************************************/
/*-Vertical command arrays---------------------------------------------------*/
static ViString cmdVertCoupling[] = {"AC","DC","GRO"};       
static ViString cmdPolarity[] = {"INV","NORM"};
static ViString cmdBand[] = {"MIN","MAX"};
static ViString cmdProbe[] = {"X1","X10","X100"};       
static ViString cmdVoltDiv[] = {"16mV","40mV","80mV", "160mV","400mV","800mV",
                        "1.6V","4V","8V", "16V","40V","80V", "160V","400V","800V",
                        "1.6kV","4kV","8kV"}; 
static ViReal64 cmdOffsetMax[] = {80E-3,80E-3,400E-3,400E-3,2.0,2.0,2.5,12.5,12.5,
                        64.0,64.0,64.0,640.0,640.0,640.0,6400.0,6400.0,6400.0}; 
static ViString cmdInpSelect[]  = {"CHAN","MEM"};   

/*-Horizontal command arrays-------------------------------------------------*/
static ViString cmdTimeDiv[] = {"20E-12","50E-12","100E-12","200E-12","500E-12",
                        "1E-9","2E-9","5E-9","10E-9","20E-9","50E-9","100E-9",
                        "200E-9","500E-9","1E-6","2E-6","5E-6","10E-6","20E-6",
                        "50E-6","100E-6","200E-6","500E-6","1E-3","2E-3","5E-3",
                        "10E-3","20E-3","50E-3","100E-3","200E-3","500E-3","1",
                        "2","5"};
static ViString cmdPretrig[] = {"0","10","20","30","40","50","60","70","80","90","100"};
static ViString cmdAcqSize[] = {"1000","2000","5000","10000"};  
static ViString cmdDispTrace[] = {"A","B","A_B","XY","A_XY"};
static ViString cmdRoll[] = {"OFF","ON","FREE"};

/*-Trigger command arrays----------------------------------------------------*/
static ViString cmdTrigCoupling[] = {"AC","DC","ON"};       
static ViString cmdSlope[] = {"NEG","POS"};
static ViString cmdSource[]= {"","INT1","INT2","INT3","INT4","LINE"};
static ViString cmdVideoField[] = {"ODD","EVEN"};

/*-Display command arrays----------------------------------------------------*/
static ViString cmdDisplayMenu[] = {"CLE","DISP","HORIZ","INT1","INT2","INT3","INT4",
                        "TRIGCOUP","TRIGMOD","TRIGSEL","CURS","TIMEMEAS","VOLTMEAS",
                        "FUNC","UTIL","FILE","PORT","HCOPY","CAL","SETUP","MEM"};       
static ViString cmdDisplayLang[] = {"FRA","ENG","DEU"};       
static ViString cmdMode[] = {"NORM","DOTJ","PERS","MINM","ENV"};       
static ViString cmdPersistence[] = {"200E-3","500E-3","1","2","5","10","INF"};       
static ViString cmdAverage[] = {"1","2","4","8","16","32","64"};       
static ViString cmdInterpolation[] = {"NONE","LIN","SPL"};   
static ViString cmdFunction[] = {"","+","-","*"};
static ViString cmdDisplayMeasure[] = {"AC","AMPL","PTP","VOLTDC","HIGH","LOW","MAX",
                        "MIN","OVER","FREQ","PER","FTIME","RTIME","PWID","NWID","PDUT"};  
/*-Hardcopy command arrays---------------------------------------------------*/
static ViString cmdDestination[] = {"","SER","CENT","GPIB"};       
static ViString cmdHardcopyLang[] = {"EFX","EESC","PCL","HPGL","DPU","POST","BMP"}; 

/*-Read command arrays-------------------------------------------------------*/
static ViString cmdReadMeasure[] = {"AC","AMPL","PTP","VOLT","HIGH","LOW","MAX","MIN",
                        "RISE:OVER","FREQ","PER","FTIME","RTIME","PWID","NWID","PDUT"};  
static ViString cmdDrive[] = {"","M:","K:"};    
static ViReal64 cmdTimeBase[] = {0.0,2E-9,5E-9,10E-9,20E-9,50E-9,100E-9,200E-9,500E-9,
                        1E-6,2E-6,5E-6,10E-6,20E-6,50E-6,100E-6,200E-6,500E-6,1E-3,
                        2E-3,5E-3,10E-3,20E-3,50E-3,100E-3,200E-3,500E-3,1.0,2.0,5.0,
                        10.0,20.0,50.0};
static ViReal64 cmdVolt[] = {2E-3,5E-3,10E-3,20E-3,50E-3,100E-3,200E-3,500E-3,1.0,
                        2.0,5.0,10.0};
static ViInt16 cmdSize[] = {1000,2000,5000,10000};

/*-Write command arrays------------------------------------------------------*/
static ViReal64 cmdVoltSense[] = {2E-3,5E-3,10E-3,20E-3,50E-3,100E-3,200E-3,500E-3,
                        1.0,2.0,5.0,10.0,20.0,50.0,100.0,200.0,500.0,1000.0};
static ViUInt16 cmdVoltMantissa[] = {4000,1000,2000};
static ViByte cmdVoltExponent[] = {7,8,8,8,9,9,9,10,10,10,11,11,11,12,12,12,13,13};
static ViUInt16 cmdTimeMantissa[] = {4000,1000,2000};
static ViByte cmdTimeExponent[] = {1,2,2,2,3,3,3,4,4,4,5,5,5,6,6,6,7,7,7,8,8,8,9,9,9,
                        10,10,10,11,11,11,12};
                                     
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT STATUS/RANGE STRUCTURE  ============================*/
/*****************************************************************************/
/* mxOX2000_stringValPair is used in the mxOX2000_errorMessage function          */
/*===============================================================================*/
typedef struct  mxOX2000_stringValPair
{
   ViStatus stringVal;
   ViPString stringName;
}  mxOX2000_tStringValPair;

struct mxOX2000_statusDataRanges {
    ViInt16 triggerMode;
    ViInt16 val2;
    ViInt16 val3;
    ViChar instrDriverRevision[256];
    ViInt16 instrMode;
};
typedef struct mxOX2000_statusDataRanges *mxOX2000_instrRange;

typedef struct {
    ViByte name[16];
    ViInt16 offsetSoft;
    ViInt16 offsetHard;
    ViUInt16 invertState;
    ViUInt16 timeBase;
    ViInt16 timeVernier;
    ViUInt16 sample;
    ViUInt16 voltSense;
    ViInt16 voltVernier;
    ViUInt32 runDelay;
    ViUInt16 coupling;
    ViUInt16 probe;
    ViUInt16 bandwidth;
    ViUInt16 mantisseTime;
    ViUInt16 expTime;
    ViUInt16 mantisseVolt;
    ViUInt16 expVolt;
    ViByte trigAndSize;
    ViByte displayMode;
    ViUInt16 storageIndex;
    ViUInt16 vernier[500];
    } OX2000Trace;
    
OX2000Trace trace;  
OX2000Trace *ptrTrace = &trace;
    
struct {
    ViReal64 offsetValue;
    ViReal64 timeBaseValue;
    ViReal64 voltSenseValue;
    ViInt16 sizeValue;
    ViByte pretrigValue;
    ViByte couplingValue;
    ViByte probeValue;
    ViByte bandwidthValue;
    }traceInfo;

ViPString triggerType = "EDGE";
ViUInt16 interface;

/*****************************************************************************/
/*= UTILITY ROUTINE DECLARATIONS (Non-Exportable Functions) =================*/
/*****************************************************************************/
ViBoolean mxOX2000_invalidViBooleanRange (ViBoolean val);
ViBoolean mxOX2000_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max);
ViBoolean mxOX2000_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max);
ViBoolean mxOX2000_invalidViUInt16Range (ViUInt16 val, ViUInt16 min, ViUInt16 max);
ViBoolean mxOX2000_invalidViUInt32Range (ViUInt32 val, ViUInt32 min, ViUInt32 max);
ViBoolean mxOX2000_invalidViReal32Range (ViReal32 val, ViReal32 min, ViReal32 max);
ViBoolean mxOX2000_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max);
ViStatus mxOX2000_makeDataReadable (ViByte tabByte[],ViBoolean interpolation, 
                                ViReal64 _VI_FAR waveformArray[], ViPInt32 numberOfSamples,
                                ViPReal64 XStart, ViPReal64 XIncrement);
ViStatus mxOX2000_encodeArray (ViByte ptrByte[],ViReal64 _VI_FAR waveformArray[],ViInt16 arraySize,
                                ViInt16 voltSense,ViReal64 offset, ViInt16 timeBase);
ViBoolean mxOX2000_invalidFilename (ViString filename, ViInt16 drive, ViString type);
ViStatus mxOX2000_instrStatus (ViSession instrSession);
ViStatus mxOX2000_initCleanUp (ViSession openRMSession, ViPSession openInstrSession, ViStatus currentStatus);
ViStatus mxOX2000_defaultInstrSetup (ViSession openInstrSession);

/*****************************************************************************/
/*====== USER-CALLABLE FUNCTIONS (Exportable Functions) =====================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Initialize                                                      */
/* Purpose:  This function opens the instrument, queries the instrument      */
/*           for its ID, and initializes the instrument to a known state.    */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_init (ViRsrc resourceName, ViBoolean IDQuery,
                    ViBoolean reset, ViPSession instrSession)
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViSession rmSession = 0;
    ViUInt32 retCnt = 0;
    ViByte rdBuffer[BUFFER_SIZE];
    ViByte model[2];
    
    /*- Check input parameter ranges ----------------------------------------*/

    if (mxOX2000_invalidViBooleanRange (IDQuery))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViBooleanRange (reset))
        return VI_ERROR_PARAMETER3;

    /*- Open instrument session ---------------------------------------------*/

    if ((mxOX2000_status = viOpenDefaultRM (&rmSession)) < 0)
        return mxOX2000_status;

    if ((mxOX2000_status = viOpen (rmSession, resourceName, VI_NULL, VI_NULL, instrSession)) < 0) {
        viClose (rmSession);
        return mxOX2000_status;
    }

    viGetAttribute (*instrSession, VI_ATTR_INTF_TYPE, &interface);
    /* Serial Interface initialized to 19200 bauds, 8 databits, 1 stopbit, no parity, RTS/CTS protocol*/
    if(interface==VI_INTF_ASRL) {
        if ((mxOX2000_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_BAUD, 19200)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
        if ((mxOX2000_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_DATA_BITS, 8)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
        if ((mxOX2000_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_STOP_BITS, VI_ASRL_STOP_ONE)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
        if ((mxOX2000_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_PARITY, VI_ASRL_PAR_NONE)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
        if ((mxOX2000_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_FLOW_CNTRL, VI_ASRL_FLOW_RTS_CTS)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
        if ((mxOX2000_status = viSetBuf (*instrSession, VI_ASRL_IN_BUF|VI_ASRL_OUT_BUF, 5000)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
        if ((mxOX2000_status = viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, 0x0D)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
        if ((mxOX2000_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
        if ((mxOX2000_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_END_OUT, VI_ASRL_END_NONE)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
        if ((mxOX2000_status = viWrite (*instrSession, "\n", 1, &retCnt)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
        if ((mxOX2000_status = viWrite (*instrSession, "SYST:COMM:SER:PACE NONE\n", 24, &retCnt)) < 0)
            return mxOX2000_status;
    }

    /*- Configure VISA Formatted I/O ----------------------------------------*/
    if ((mxOX2000_status = viSetAttribute (*instrSession, VI_ATTR_TERMCHAR_EN, VI_FALSE)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
    if ((mxOX2000_status = viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
    if ((mxOX2000_status = viSetBuf (*instrSession, VI_READ_BUF|VI_WRITE_BUF, 4000)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
    if ((mxOX2000_status = viSetAttribute (*instrSession, VI_ATTR_WR_BUF_OPER_MODE,VI_FLUSH_ON_ACCESS)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
    if ((mxOX2000_status = viSetAttribute (*instrSession, VI_ATTR_RD_BUF_OPER_MODE,VI_FLUSH_ON_ACCESS)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);

    /*- Identification Query ------------------------------------------------*/
    if (IDQuery) {
        if ((mxOX2000_status = viWrite (*instrSession, "*IDN?\n", 6, &retCnt)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
        if ((mxOX2000_status = viRead (*instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
            return mxOX2000_status;
        Scan (rdBuffer,"%s>%s[t44d]%c[d]%s[t44d]%s[w2d]%i[b1d]%c[d]%i[b1d]%c",model);
        if (model[0] != 'a')
            return mxOX2000_initCleanUp (rmSession, instrSession, VI_ERROR_FAIL_ID_QUERY);
    }

    /*- Reset instrument ----------------------------------------------------*/
    if (reset) {
        if ((mxOX2000_status = mxOX2000_reset (*instrSession)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
    }       
    else  /*- Send Default Instrument Setup ---------------------------------*/
        {
        if ((mxOX2000_status = mxOX2000_defaultInstrSetup (*instrSession)) < 0)
            return mxOX2000_initCleanUp (rmSession, instrSession, mxOX2000_status);
        }
        
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(*instrSession)) < 0)
        return mxOX2000_status;

    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Example                                                         */
/* Purpose:  This function is an example of this driver's functions use.     */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_example (ViSession instrSession, ViInt16 channel,
                    ViInt16 timebase, ViInt16 voltageRange, ViInt16 coupling, 
                    ViReal64 _VI_FAR data[], ViPInt32 numberOfSamples, 
                    ViPReal64 XStart, ViPReal64 XIncrement)
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    
    /*- Check input parameter ranges ----------------------------------------*/    
    if (mxOX2000_invalidViInt16Range (channel, 1, 4))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range (timebase, 0, 31))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViInt16Range (voltageRange, 0, 11))
        return VI_ERROR_PARAMETER4;
    if (mxOX2000_invalidViInt16Range (coupling, 0, 2))
        return VI_ERROR_PARAMETER5;
    
    /*- Resetting the instrument into a known state -------------------------*/
    if((mxOX2000_status = mxOX2000_reset (instrSession)) < 0)
        return mxOX2000_status;

    /*- Configuring vertical settings----------------------------------------*/
    if((mxOX2000_status = mxOX2000_verticalSetup (instrSession, channel, VI_ON, coupling, 0, VI_ON, voltageRange, 0.0, VI_ON)) < 0)
        return mxOX2000_status;
        
    /*- Setting timebase ----------------------------------------------------*/
    if((mxOX2000_status = mxOX2000_horizontalSetup (instrSession, 0, 5, 0.0, timebase, 0.0, timebase, 0.0)) < 0)
        return mxOX2000_status;

    /*- Setting trigger mode to AUTO-----------------------------------------*/
    /*- OX2000 starts acquiring data immediately after it is triggered-------*/
    if((mxOX2000_status = mxOX2000_setGeneralTrigger (instrSession, channel, VI_ON, 0, 0, 0, 0, 0.0, 80.0E-9)) < 0)
        return mxOX2000_status;
        
    /*- Sending single-shot trigger in wait mode-----------------------------*/
    /*- This function will return as long as the acquisition is running------*/
    if((mxOX2000_status = mxOX2000_runAcquisition (instrSession, 1, VI_ON)) < 0)
        return mxOX2000_status;
        
    /*- Reads trace from the instrument--------------------------------------*/ 
    if((mxOX2000_status = mxOX2000_readChToArr (instrSession, channel, VI_OFF, data, numberOfSamples, XStart, XIncrement)) < 0)
        return mxOX2000_status; 
                   
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Configure OX2000 Vertical Settings                              */
/* Purpose:  This function configures the vertical settings for a specified  */
/*           channel                                                         */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_verticalSetup (ViSession instrSession, ViInt16 input,
                    ViBoolean state, ViInt16 coupling, ViInt16 probe, ViBoolean polarity, 
                    ViInt16 sense, ViReal64 offset, ViBoolean band)
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViByte wrBuffer[BUFFER_SIZE];
    ViInt32 retCnt;

    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(input,1,4))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViBooleanRange(state))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViInt16Range(coupling,0,2))
        return VI_ERROR_PARAMETER4;
    if (mxOX2000_invalidViInt16Range(probe,0,2))
        return VI_ERROR_PARAMETER5;
    if (mxOX2000_invalidViBooleanRange(polarity))
        return VI_ERROR_PARAMETER6;
    if (mxOX2000_invalidViInt16Range(sense,0,17))
        return VI_ERROR_PARAMETER7;
    switch (probe) {
        /*--Probe x1---------------------------------------------------------*/ 
        case 0 :    if (mxOX2000_invalidViInt16Range(sense,0,11))
                        return VI_ERROR_OX2000_INCONSISTENT_PARAMETER7;
                    break;
        /*--Probe x10--------------------------------------------------------*/ 
        case 1 :    if (mxOX2000_invalidViInt16Range(sense,3,14))
                        return VI_ERROR_OX2000_INCONSISTENT_PARAMETER7; 
                    break;
        /*--Probe x100-------------------------------------------------------*/ 
        case 2 :    if (mxOX2000_invalidViInt16Range(sense,6,17))
                        return VI_ERROR_OX2000_INCONSISTENT_PARAMETER7; 
                    break;
    }      
    if (mxOX2000_invalidViReal64Range(offset,-cmdOffsetMax[sense],cmdOffsetMax[sense]))
        return VI_ERROR_PARAMETER8;
    if (mxOX2000_invalidViBooleanRange(band))
        return VI_ERROR_INSTR_PARAMETER9;
    
    /*-Send vertical config command------------------------------------------*/
    Fmt(wrBuffer,"INP%i[b2]:SEL CHAN;STATE %i[b2];COUP %s;POL %s;PROB %s\n",
        input,state,cmdVertCoupling[coupling],cmdPolarity[polarity],cmdProbe[probe]);
    if ((mxOX2000_status = viWrite (instrSession,wrBuffer,StringLength(wrBuffer),&retCnt)) < 0)
        return mxOX2000_status; 

    Fmt(wrBuffer,"SENS:BAND %s;VOLT%i[b2]:PTP %s;OFFS %f\n",cmdBand[band],input,cmdVoltDiv[sense],offset);
    if ((mxOX2000_status = viWrite (instrSession,wrBuffer,StringLength(wrBuffer),&retCnt)) < 0)
        return mxOX2000_status; 
            
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Configure OX2000 Horizontal Settings                            */
/* Purpose:  This function configures the horizontal settings of the         */
/*           instrument                                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_horizontalSetup (ViSession instrSession,ViInt16 size, 
                    ViInt16 pretrig, ViReal64 delay, ViInt16 rangeA, ViReal64 delayA, 
                    ViInt16 rangeB,ViReal64 delayB)
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViReal64 convBuf;
    
    /*-Check input parameter ranges------------------------------------------*/
    /*-acquisition size, run after delay, pretrig, time bases A & B----------*/
    if (mxOX2000_invalidViInt16Range(size,0,3))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range(pretrig,0,10))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViInt16Range(rangeA,0,31))
        return VI_ERROR_PARAMETER5;
    if (mxOX2000_invalidViInt16Range(rangeB,0,34))
        return VI_ERROR_PARAMETER7;
    if (rangeA < 7) {
        if (mxOX2000_invalidViInt16Range(rangeB,0,(rangeA + size)))    
            return VI_ERROR_OX2000_INCONSISTENT_PARAMETER7;
    }
    else {
        if (mxOX2000_invalidViInt16Range(rangeB,(rangeA - 6),(rangeA + size)))    
            return VI_ERROR_OX2000_INCONSISTENT_PARAMETER7;
    }
    Fmt(&convBuf,"%f<%s",cmdTimeDiv[rangeA]);
    if (mxOX2000_invalidViReal64Range(delay,0,(4000000.0 * convBuf)))
        return VI_ERROR_PARAMETER4;

    /*-Send acquisition size, run after delay, pretrig, time bases A & B-----*/
    if ((mxOX2000_status = viPrintf (instrSession,"SENS:SWE:POIN %s;OREF:LOC %s\n",cmdAcqSize[size],cmdPretrig[pretrig])) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viPrintf (instrSession,"SENS:SWE:OFFS:TIME %lf\n",delay)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viPrintf (instrSession,"SENS:SWE1:TINT %s\n",cmdTimeDiv[rangeA])) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viPrintf (instrSession,"SENS:SWE2:TINT %s\n",cmdTimeDiv[rangeB])) < 0)
        return mxOX2000_status;

        
    /*-Check input parameter ranges (delays A & B)---------------------------*/
    if ((mxOX2000_status = viPrintf (instrSession,"SENS:SWE1:SCRE:TIME? MIN\n")) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viScanf (instrSession,"%lf",&convBuf)) < 0)
        return mxOX2000_status;
    if (mxOX2000_invalidViReal64Range(delayA,convBuf,-convBuf))
        return VI_ERROR_PARAMETER6;
    
    if ((mxOX2000_status = viPrintf (instrSession,"SENS:SWE2:SCRE:TIME? MIN\n")) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viScanf (instrSession,"%lf",&convBuf)) < 0)
        return mxOX2000_status;
    if (mxOX2000_invalidViReal64Range(delayB,convBuf,-convBuf))
        return VI_ERROR_PARAMETER8;
        
    /*-Send delays A & B-----------------------------------------------------*/
    if ((mxOX2000_status = viPrintf (instrSession,"SENS:SWE1:SCRE:TIME %lf\n",delayA)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viPrintf (instrSession,"SENS:SWE2:SCRE:TIME %lf\n",delayB)) < 0)
        return mxOX2000_status;

    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Configure OX2000 Horizontal Display                             */
/* Purpose:  This function configures the horizontal display                 */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_horizontalDisplay (ViSession instrSession, ViInt16 trace,
                    ViInt16 x, ViInt16 y, ViInt16 roll,ViBoolean cumul)
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViByte wrBuffer[BUFFER_SIZE];
    ViInt32 retCnt;

    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(trace,0,4))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range(x,1,4))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViInt16Range(y,1,4))
        return VI_ERROR_PARAMETER4;
    if (mxOX2000_invalidViInt16Range(roll,0,2))
        return VI_ERROR_PARAMETER5;
    if (mxOX2000_invalidViBooleanRange(cumul))
        return VI_ERROR_PARAMETER6;

    Fmt (wrBuffer,"DISP:TRAC:FORM %s;XY:XDEF INT%i[b2];YDEF INT%i[b2]\n",cmdDispTrace[trace],x,y);
    if ((mxOX2000_status = viWrite (instrSession,wrBuffer,StringLength(wrBuffer),&retCnt)) < 0)
        return mxOX2000_status;
    Fmt (wrBuffer,"DISP:TRAC:ROLL %s;SINGLE:CUMUL %i[b2]\n",cmdRoll[roll], cumul);
    if ((mxOX2000_status = viWrite (instrSession,wrBuffer,StringLength(wrBuffer),&retCnt)) < 0)
        return mxOX2000_status;
        
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: General Trigger                                                 */
/* Purpose:  This function sets general parameters for trigger               */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_setGeneralTrigger (ViSession instrSession,ViInt16 channel,
                                        ViBoolean slope,ViInt16 coupling,ViBoolean hfReject,
                                        ViBoolean noiseReject,ViInt16 type,ViReal64 level,ViReal64 hold)
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViByte wrBuffer[BUFFER_SIZE];
    ViReal64 convBuf;

    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(channel,1,4))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViBooleanRange(slope))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViInt16Range(coupling,0,2))
        return VI_ERROR_PARAMETER4;
    if (mxOX2000_invalidViBooleanRange(hfReject))
        return VI_ERROR_PARAMETER5;
    if (mxOX2000_invalidViBooleanRange(noiseReject))
        return VI_ERROR_PARAMETER6;
    if (mxOX2000_invalidViInt16Range(type,0,2))
        return VI_ERROR_PARAMETER7;
    if ((mxOX2000_status = viPrintf (instrSession,"SENS:VOLT%hd:PTP?\n",channel)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viScanf (instrSession,"%lf",&convBuf)) < 0)
        return mxOX2000_status;
    if (mxOX2000_invalidViReal64Range(level,-1.5 * convBuf,1.5 * convBuf))
        return VI_ERROR_PARAMETER8;
    if (mxOX2000_invalidViReal64Range(hold,80E-9,1.0))
        return VI_ERROR_INSTR_PARAMETER9;
        
    /*-Set slope,coupling,level,holdoff and filter parameters----------------*/
    if ((mxOX2000_status = viPrintf (instrSession,"TRIG:SOUR INT%hd;SLOPE %s;LEVEL %lf;FILT:NOISE %hd;LPASS %hd\n",
                channel,cmdSlope[slope],level,noiseReject,hfReject)) < 0)
        return mxOX2000_status;
    if (coupling < 2) {
        if ((mxOX2000_status = viPrintf (instrSession,"TRIG:COUP %s\n",cmdTrigCoupling[coupling])) < 0)
            return mxOX2000_status;
    }                                            
    else {
        if (hfReject)  
            return VI_ERROR_OX2000_INCONSISTENT_PARAMETER5;
        if ((mxOX2000_status = viPrintf (instrSession,"TRIG:FILT:HPASS %s\n",
                cmdTrigCoupling[coupling])) < 0)
            return mxOX2000_status;
    }
    Fmt (wrBuffer,"TRIG:HOLD %f\n",hold);
    if ((mxOX2000_status = viPrintf (instrSession,"%s",wrBuffer)) < 0)
        return mxOX2000_status;
        
    /*-Set trigger type------------------------------------------------------*/
    switch (type) {
        case 0 :    if ((mxOX2000_status = viPrintf (instrSession,"TRIG:ATR ON\n")) < 0)
                        return mxOX2000_status;
                    break;
        case 1 :    if ((mxOX2000_status = viPrintf (instrSession,"TRIG:ATR ON;ATR OFF\n")) < 0)
                        return mxOX2000_status;
                    break;
        case 2 :    if ((mxOX2000_status = viPrintf (instrSession,"TRIG:LEVEL:AUTO ON\n")) < 0)
                        return mxOX2000_status;
                    break;
    }
        
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Edge Trigger                                                    */
/* Purpose:  This function sets edge trigger type                            */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_setEdgeTrigger (ViSession instrSession, ViInt16 source,
                    ViBoolean window, ViReal64 mainLevel, ViReal64 auxLevel)
{                                
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViReal64 convBuf;
    ViByte rdBuffer[BUFFER_SIZE];
    ViInt32 retCnt;
    ViInt16 tstRes;
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(source,1,5))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViBooleanRange(window))
        return VI_ERROR_PARAMETER3;
        
    /*-If trigger source is not Line-----------------------------------------*/
    if (source < 5) {
        if ((mxOX2000_status = viPrintf (instrSession,"SENS:VOLT%hd:PTP?\n",source)) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viScanf (instrSession,"%lf",&convBuf)) < 0)
            return mxOX2000_status;
        if (mxOX2000_invalidViReal64Range(mainLevel,-1.5 * convBuf,1.5 * convBuf))
            return VI_ERROR_PARAMETER4;
        if (mxOX2000_invalidViReal64Range(auxLevel,-1.5 * convBuf,1.5 * convBuf))
            return VI_ERROR_PARAMETER5;
            
        /*-Set source and levels for edge trigger----------------------------*/
        if ((mxOX2000_status = viPrintf (instrSession,"TRIG:TYPE EDGE;SOUR INT%hd\n", source)) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viPrintf (instrSession,"TRIG:LEV %lf;LEV:WIND %hd;WIND:AUXLEV %lf\n",mainLevel,window,auxLevel)) < 0)
            return mxOX2000_status;
    }
    
    /*-If trigger source is Line---------------------------------------------*/
    else {
        if ((mxOX2000_status = viPrintf (instrSession,"TRIG:TYPE EDGE;SOUR LINE\n")) < 0)
            return mxOX2000_status;
        if (window) return VI_ERROR_OX2000_INCONSISTENT_PARAMETER3;
    }   
    
    triggerType = "EDGE";
    
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Delayed Trigger                                                 */
/* Purpose:  This function sets delayed trigger type                         */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_setDelayedTrigger (ViSession instrSession, ViInt16 source,
                    ViInt16 qualifier,ViReal64 delay)
{                                
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViByte wrBuffer[BUFFER_SIZE];
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(source,1,5))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range(qualifier,1,5))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViReal64Range(delay,40.0E-9,16.0))
        return VI_ERROR_PARAMETER4;
        
    /*-Set delay parameters--------------------------------------------------*/
    Fmt(wrBuffer,"ARM:SEQ2:SOUR %s;DELAY %f[p9]\n",cmdSource[qualifier],delay);
    if ((mxOX2000_status = viPrintf (instrSession,"%s", wrBuffer)) < 0)  
        return mxOX2000_status;
    if ((mxOX2000_status = viPrintf (instrSession,"TRIG:SEQ2:SOUR %s\n",cmdSource[source])) < 0)
        return mxOX2000_status;
        
    triggerType = "DELAY";
    
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Count Trigger                                                   */
/* Purpose:  This function sets event-count trigger type                     */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_setCountTrigger (ViSession instrSession, ViBoolean source,
                    ViInt16 qualifier, ViInt16 counter, ViInt32 events)
{                                
    ViStatus mxOX2000_status = VI_SUCCESS;
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViBooleanRange(source))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range(qualifier,1,5))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViInt16Range(counter,1,5))
        return VI_ERROR_PARAMETER4;
    if (mxOX2000_invalidViInt32Range(events,1,16777215))
        return VI_ERROR_PARAMETER5;
        
    /*-Set count trigger parameters------------------------------------------*/
    if ((mxOX2000_status = viPrintf (instrSession,"ARM:SEQ3:LAYER2:SOUR %s\n",cmdSource[qualifier])) < 0)
        return mxOX2000_status;

    if ((mxOX2000_status = viPrintf (instrSession,"ARM:SEQ3:SOUR %s;ECOUNT %ld\n",cmdSource[counter],events)) < 0)
        return mxOX2000_status;
    
    /*-Select trigger source-------------------------------------------------*/
    if (source) {
        if ((mxOX2000_status = viPrintf (instrSession,"TRIG:SEQ3:SOUR %s\n",cmdSource[counter])) < 0)
            return mxOX2000_status;
    }
    else {
        if ((mxOX2000_status = viPrintf (instrSession,"TRIG:SEQ3:SOUR %s\n",cmdSource[qualifier])) < 0)
            return mxOX2000_status;
    }
        
    triggerType = "EVENT";
    
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Video Trigger                                                   */
/* Purpose:  This function sets video trigger type                           */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_setVideoTrigger (ViSession instrSession, ViInt16 source,
                    ViBoolean standard, ViBoolean polarity, ViBoolean field,
                    ViInt16 line)
{                                
    ViStatus mxOX2000_status = VI_SUCCESS;
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(source,1,4))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViBooleanRange(standard))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViBooleanRange(polarity))
        return VI_ERROR_PARAMETER4;
    if (mxOX2000_invalidViBooleanRange(field))
        return VI_ERROR_PARAMETER5;
    if (mxOX2000_invalidViInt16Range(line,1,625))
        return VI_ERROR_PARAMETER6;
    
    switch (standard) {
        case 0 :    if (field) {
                        if (mxOX2000_invalidViInt16Range(line,1,262))
                            return VI_ERROR_OX2000_INCONSISTENT_PARAMETER6;
                    }
                    else 
                        if (mxOX2000_invalidViInt16Range(line,1,263))
                            return VI_ERROR_OX2000_INCONSISTENT_PARAMETER6;
                    break;
        case 1 :    if (field) {
                        if (mxOX2000_invalidViInt16Range(line,314,625))
                            return VI_ERROR_OX2000_INCONSISTENT_PARAMETER6;
                    }
                    else 
                        if (mxOX2000_invalidViInt16Range(line,1,313))
                            return VI_ERROR_OX2000_INCONSISTENT_PARAMETER6;
                    break;
    }       
        

    if ((mxOX2000_status = viPrintf (instrSession,"TRIG:TYPE VIDEO;SOUR INT%hd;VIDEO:SSIG %s;FIELD:SEL %s;FORM:LPFR %hd\n",
            source,cmdSlope[polarity],cmdVideoField[field],525 + 100 * standard)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viPrintf (instrSession,"TRIG:VID:LINE %hd\n",line)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viPrintf (instrSession,"TRIG:LEVEL:AUTO OFF;:TRIG:SEQ1:ATR OFF\n")) < 0)
        return mxOX2000_status;

    triggerType= "EDGE";
    
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}
/*===========================================================================*/
/* Function: Display Options                                                 */
/* Purpose:  This function sets display options.                             */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_setOptionDisplay (ViSession instrSession, ViBoolean grid,
                    ViInt16 color,ViBoolean screen, ViInt16 menu, ViInt16 language)
{
    ViStatus mxOX2000_status = VI_SUCCESS;

    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViBooleanRange(grid))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range(color,0,2))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViBooleanRange(screen))
        return VI_ERROR_PARAMETER4;
    if (mxOX2000_invalidViInt16Range(menu,0,20))
        return VI_ERROR_PARAMETER5;
    if (mxOX2000_invalidViInt16Range(language,0,2))
        return VI_ERROR_PARAMETER6;

    /*-Set option display parameters-----------------------------------------*/
    if ((mxOX2000_status = viPrintf (instrSession,"DISP:LANG %s;TRAC:NSCRE %hd;GRAT:COL %hd;GRID %hd\n",
                    cmdDisplayLang[language],screen * 3 + 1,color,grid)) < 0)
        return mxOX2000_status;
    if (!menu) {               
        if ((mxOX2000_status = viPrintf (instrSession,"DISP:CURS:CLEAR;:DISP:MENU CLE\n")) < 0)
            return mxOX2000_status;
    }
    if ((mxOX2000_status = viPrintf (instrSession,"DISP:MENU %s\n",cmdDisplayMenu[menu])) < 0)
        return mxOX2000_status;

    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Display Waveform                                                */
/* Purpose:  This function sets waveform display.                            */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_setWaveformDisplay (ViSession instrSession,ViInt16 mode,
                    ViInt16 rate,ViInt16 persistence, ViInt16 interpol)
{
    ViStatus mxOX2000_status = VI_SUCCESS;

    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(mode,0,4))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range(rate,0,6))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViInt16Range(persistence,0,6))
        return VI_ERROR_PARAMETER4;
    if (mxOX2000_invalidViInt16Range(interpol,0,2))
        return VI_ERROR_PARAMETER5;

    /*-Set waveform display parameters---------------------------------------*/
    if ((mxOX2000_status = viPrintf (instrSession,"DISP:TRAC:MODE %s;PERS %s;:CALC:INT %s\n",
            cmdMode[mode],cmdPersistence[persistence],cmdInterpolation[interpol])) < 0)
        return mxOX2000_status;
    if (!rate) {
        if ((mxOX2000_status = viPrintf (instrSession,"SENS:AVER OFF\n")) < 0)
            return mxOX2000_status;
    }
    else {
        if ((mxOX2000_status = viPrintf (instrSession,"SENS:AVER ON;AVER:COUNT %s\n",cmdAverage[rate])) < 0)
            return mxOX2000_status;
    }
    
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Function                                                        */
/* Purpose:  This function sets OX2000 mathematical function.                */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_setFunction (ViSession instrSession, ViBoolean select,
                    ViInt16 func, ViInt16 source1, ViInt16 source2)
{
    ViStatus mxOX2000_status = VI_SUCCESS;

    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViBooleanRange(select))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range(func,0,3))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViInt16Range(source1,1,4))
        return VI_ERROR_PARAMETER4;
    if (mxOX2000_invalidViInt16Range(source2,1,4))
        return VI_ERROR_PARAMETER5;
        
    /*-Set mathematical function parameters----------------------------------*/
    if (!func) {
        if ((mxOX2000_status = viPrintf (instrSession,"CALC:MATH:EXPR%hd:DEL\n",select + 1)) < 0)
            return mxOX2000_status;
    }
    else {
        if ((mxOX2000_status = viPrintf (instrSession,"CALC:MATH:EXPR%hd (INT%hd%sINT%hd)\n",
                select+1,source1,cmdFunction[func],source2)) < 0)
            return mxOX2000_status;
    }
    
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Set Measurement                                                 */
/* Purpose:  This function sets OX2000 automatical measurements.             */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_setMeasurement(ViSession instrSession, ViInt16 edit,
                    ViInt16 meas, ViInt16 reference)
{                              
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViInt32 retCnt;
    ViInt16 i=0,temp=0;
    static ViByte string[1000],tabByte[MAX_MEM_SIZE];
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(edit,0,2))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range(meas,0,15))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViInt16Range(reference,1,4))
        return VI_ERROR_PARAMETER4;
        
    /*-Display/clear measurement---------------------------------------------*/
    if ((mxOX2000_status = viPrintf (instrSession,"DISP:MEAS:CAT?\n")) < 0)                        
        return mxOX2000_status;                                                                    
    if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
        return mxOX2000_status;
        
    switch (edit) {
        case 0 : Fmt(string,"%s<\"%s\",INT%i[b2]",cmdDisplayMeasure[meas],reference);
                 if ((temp = FindPattern (tabByte, 0, -1, string, 0, 0)) == -1) {
                     if ((mxOX2000_status = viPrintf (instrSession,"DISP:MEAS:NAME \"%s\",INT%hd\n",
                               cmdDisplayMeasure[meas],reference)) < 0)
                         return mxOX2000_status;
                 }
                 break;
        case 1 : Fmt(string,"%s<\"%s\",INT%i[b2]",cmdDisplayMeasure[meas],reference);
                 if ((temp = FindPattern (tabByte, 0, -1, string, 0, 0)) != -1) {
                     if ((mxOX2000_status = viPrintf (instrSession,"DISP:MEAS:DEL \"%s\",INT%hd\n",
                             cmdDisplayMeasure[meas],reference)) < 0)
                         return mxOX2000_status;
                 }
                 break;
        case 2 : while ((temp = FindPattern (tabByte, i, -1, ",\"", 0, 0)) != -1) {
                     Fmt(string,"%s<DISP:MEAS:DEL %s[i*w*]\n",i,temp-i,tabByte);
                     if ((mxOX2000_status = viWrite (instrSession,string,temp + 15,&retCnt)) < 0)
                         return mxOX2000_status;
                     i = temp + 1;
                  }
                  if ((temp = FindPattern (tabByte, 0, -1, "\n", 0, 0)) != 0) {
                     Fmt(string,"%s<DISP:MEAS:DEL %s[i*w*]\n",i,temp-i,tabByte);
                     if ((mxOX2000_status = viWrite (instrSession,string,temp + 15,&retCnt)) < 0)
                         return mxOX2000_status;
                  }
                  break;
    }       
    
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Set Cursors                                                     */
/* Purpose:  This function sets OX2000 cursors.                              */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_setCursors (ViSession instrSession, ViBoolean cursor,
                    ViInt16 reference, ViReal64 t1, ViReal64 t2, ViReal64 v1, 
                    ViReal64 v2)
{                              
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViReal64 convBuf;
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViBooleanRange(cursor))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range(reference,1,4))
        return VI_ERROR_PARAMETER3;
    if ((mxOX2000_status = viPrintf (instrSession,"DISP:CURS:TIME:POS? MAX\n")) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viScanf (instrSession,"%lf",&convBuf)) < 0)
        return mxOX2000_status;
    if (mxOX2000_invalidViReal64Range(t1,-convBuf,convBuf))
        return VI_ERROR_PARAMETER4;
    if (mxOX2000_invalidViReal64Range(t2,-convBuf,convBuf))
        return VI_ERROR_PARAMETER5;
    if ((mxOX2000_status = viPrintf (instrSession,"DISP:CURS:VOLT:POS? MAX\n")) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viScanf (instrSession,"%lf",&convBuf)) < 0)
        return mxOX2000_status;
    if (mxOX2000_invalidViReal64Range(v1,-convBuf,convBuf))
        return VI_ERROR_PARAMETER6;
    if (mxOX2000_invalidViReal64Range(v2,-convBuf,convBuf))
        return VI_ERROR_PARAMETER7;
        
    /*-Clear cursors---------------------------------------------------------*/
    if (!cursor) {
        if ((mxOX2000_status = viPrintf (instrSession,"DISP:CURS:CLEAR\n")) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viPrintf (instrSession,"DISP:MENU CLE\n")) < 0)
            return mxOX2000_status;
    }
    
    /*-Set cursors-----------------------------------------------------------*/
    else {
        if ((mxOX2000_status = viPrintf (instrSession,"DISP:MENU CLE\n")) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viPrintf (instrSession,"DISP:CURS:REF INT%hd\n",reference)) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viPrintf (instrSession,"DISP:CURS:TIM:POS1 %lf\n",t1)) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viPrintf (instrSession,"DISP:CURS:TIM:POS2 %lf\n",t2)) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viPrintf (instrSession,"DISP:CURS:VOLT:POS1 %lf\n",v1)) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viPrintf (instrSession,"DISP:CURS:VOLT:POS2 %lf\n",v2)) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viPrintf (instrSession,"DISP:MENU CURS\n")) < 0)
            return mxOX2000_status;
    }
        
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Hardcopy                                                        */
/* Purpose:  This function configures OX2000 hardcopy facility.              */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_setHardcopy (ViSession instrSession, ViInt16 device,
                    ViInt16 port, ViString filename, ViBoolean type, 
                    ViBoolean pageFormat,ViInt16 page, ViBoolean penStyle)
{                              
    ViStatus mxOX2000_status = VI_SUCCESS;
            
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(device,0,6))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range(port,0,3))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViBooleanRange(type))
        return VI_ERROR_PARAMETER5;
    if (mxOX2000_invalidViBooleanRange(pageFormat))
        return VI_ERROR_PARAMETER6;
    if (mxOX2000_invalidViInt16Range(page,1,4))
        return VI_ERROR_PARAMETER7;
    if (mxOX2000_invalidViBooleanRange(penStyle))
        return VI_ERROR_PARAMETER8;
        
    /*-Set hardcopy parameters-----------------------------------------------*/
    if (!port) {
        if (StringLength(filename) != 8) 
            return VI_ERROR_PARAMETER4;
        if ((mxOX2000_status = viPrintf (instrSession,"HCOP:DEST \"%s\"\n",filename)) < 0)
            return mxOX2000_status;
    }
    else {
        if ((mxOX2000_status = viPrintf (instrSession,"HCOP:DEST \"%s\"\n",cmdDestination[port])) < 0)
            return mxOX2000_status;
    }
    if ((mxOX2000_status = viPrintf (instrSession,"HCOP:DEV:LANG %s;PEN %hd;COL %hd\n",
            cmdHardcopyLang[device],2 + 5 * penStyle,type)) < 0)
        return mxOX2000_status;
    if ((device != 3) && pageFormat)  
        return VI_ERROR_OX2000_INCONSISTENT_PARAMETER6;
    if (pageFormat) {
        if ((mxOX2000_status = viPrintf (instrSession,"HCOPY:PAGE:DIM:QUAD %hd\n",page)) < 0)
            return mxOX2000_status;
    }
        
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Run Autoset                                                     */
/* Purpose:  This function performs an automatic setup of the OX2000.        */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_runAutoset (ViSession instrSession)
{                              
    ViStatus mxOX2000_status = VI_SUCCESS;
    
    ViInt16 i,state;
    
    for (i=1;i<5;i++) {
        if ((mxOX2000_status = viPrintf (instrSession,"INP%hd?\n",i)) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viScanf (instrSession,"%hd",&state)) < 0)
            return mxOX2000_status;
        if (state) i = 5;
    }
    if (i==5)  
        return VI_ERROR_OX2000_NO_CHAN_ENABLED;
            
    if ((mxOX2000_status = viPrintf (instrSession,"AUTOS:EXE\n")) < 0)
        return mxOX2000_status;
        
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Run Acquisition                                                 */
/* Purpose:  This function performs an acquisition session.                  */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_runAcquisition (ViSession instrSession, ViInt16 mode,
                    ViBoolean wait)
{                              
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViInt32 retCnt;
    ViByte rdBuffer[BUFFER_SIZE];
    ViByte currentTriggerType[6];
    ViByte currentTriggerState;
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(mode,0,2))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViBooleanRange(wait))
        return VI_ERROR_PARAMETER3;
    
    /*-Get current trigger mode----------------------------------------------*/
    if ((mxOX2000_status = viPrintf (instrSession,"INIT:CONT:NAME?\n")) < 0)                           
        return mxOX2000_status;                                                                        
    if ((mxOX2000_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        return mxOX2000_status;
    if (Scan (rdBuffer, "%s>%s[t#]%i[b1]",currentTriggerType, &currentTriggerState) != 2)
        return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    
    switch(mode) {
        case 0  :   if ((mxOX2000_status = viPrintf (instrSession,"INIT:CONT:NAME %s,1\n",triggerType)) < 0)
                        return mxOX2000_status;
                    break;
        case 1  :   if ((mxOX2000_status = viPrintf (instrSession,"INIT:CONT:NAME %s0\n",currentTriggerType)) < 0)
                        return mxOX2000_status;
                    if (wait) {
                        if ((mxOX2000_status = viPrintf (instrSession,"INIT:NAME %s;:*WAI\n",triggerType)) < 0)
                            return mxOX2000_status;
                    }
                    else {
                        if ((mxOX2000_status = viPrintf (instrSession,"INIT:NAME %s\n",triggerType)) < 0)
                            return mxOX2000_status;
                    }
                    break;
        case 2  :   if ((mxOX2000_status = viPrintf (instrSession,"ABORT\n")) < 0)
                        return mxOX2000_status;
                    break;
    }
    
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Run Hardcopy                                                    */
/* Purpose:  This function performs a hardcopy.                              */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_runHardcopy (ViSession instrSession, ViBoolean action)
{                              
    ViStatus mxOX2000_status = VI_SUCCESS;
    
    if (action) {               
        if ((mxOX2000_status = viPrintf (instrSession,"HCOP:SDUMP\n")) < 0)
            return mxOX2000_status;
    }
    else {
        if ((mxOX2000_status = viPrintf (instrSession,"HCOP:SDUMP:UNDO\n")) < 0)
            return mxOX2000_status;
    }   
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Read Measurement                                                */
/* Purpose:  This function reads a specified measurement.                    */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_readMeasurement (ViSession instrSession, ViInt16 measure,
                    ViInt16 reference, ViPReal64 reading)
{                              
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViByte rdBuffer[BUFFER_SIZE];
    ViInt32 retCnt;
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(measure,0,15))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range(reference,1,4))
        return VI_ERROR_PARAMETER3;
        
    if ((mxOX2000_status = viPrintf (instrSession,"MEAS:%s? INT%hd\n",cmdReadMeasure[measure],reference)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = Scan (rdBuffer,"%s>%f",reading))!=1)
        return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
     
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Read Cursors                                                    */
/* Purpose:  This function reads OX2000 cursors.                             */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_readCursors (ViSession instrSession, ViInt16 reference,
                    ViPReal64 t1, ViPReal64 t2, ViPReal64 dt, ViPReal64 v1, 
                    ViPReal64 v2, ViPReal64 dv)
{                              
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViByte rdBuffer[BUFFER_SIZE];
    ViInt32 retCnt;
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(reference,1,4))
        return VI_ERROR_PARAMETER2;

    /*-Display cursor menu---------------------------------------------------*/
    if ((mxOX2000_status = viPrintf (instrSession,"DISP:CURS:REF INT%hd\n",reference)) < 0)
        return mxOX2000_status;
        
    /*-Set cursor positions--------------------------------------------------*/
    if ((mxOX2000_status = viPrintf (instrSession,"DISP:CURS:TIM:POS1?\n")) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viScanf (instrSession,"%lf",t1)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viPrintf (instrSession,"DISP:CURS:TIM:POS2?\n")) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viScanf (instrSession,"%lf",t2)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viPrintf (instrSession,"DISP:CURS:VOLT:POS1?\n")) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viScanf (instrSession,"%lf",v1)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viPrintf (instrSession,"DISP:CURS:VOLT:POS2?\n")) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viScanf (instrSession,"%lf",v2)) < 0)
        return mxOX2000_status;
        
    /*-Calculate 'delta t' & 'delta v'---------------------------------------*/
    *dt = *t2 - *t1;
    *dv = *v2 - *v1;
        
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Read Channel To Array                                           */
/* Purpose:  This function reads a trace from the instrument.                */
/*           Data is stored in an array                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_readChToArr (ViSession instrSession, ViInt16 source,
                    ViBoolean interpolation, ViReal64 _VI_FAR waveformArray[],
                    ViPInt32 numberOfSamples, ViPReal64 XStart, ViPReal64 XIncrement)
{   
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViInt16 headerSize;
    ViInt32 retCnt,size;
    ViByte tstRes;
    static ViByte tabByte[MAX_MEM_SIZE];
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(source,1,4))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViBooleanRange(interpolation))
        return VI_ERROR_PARAMETER3;
    
    /*-Check if source is active---------------------------------------------*/
    if ((mxOX2000_status = viPrintf (instrSession,"INP%hd:STAT?\n",source)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viScanf (instrSession,"%hd", &tstRes)) < 0)
        return mxOX2000_status;
    if (tstRes != 1) 
        return VI_ERROR_OX2000_CHAN_NOT_ACTIVE;
    
    if ((mxOX2000_status = viPrintf (instrSession,"INP%hd:SEL?\n",source)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
        return mxOX2000_status;
    if (tabByte[0] != 'C')  
        return VI_ERROR_OX2000_CHAN_NOT_ACTIVE;
        
    /*-Modify Time Out for big waveform transfers------------------------*/
    if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, 2*TMO_VALUE)) < 0)
        return mxOX2000_status;

    /*-Modify Protocol for binary file transfers-------------------------*/
    viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
    if(interface==VI_INTF_ASRL){
        if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
            return mxOX2000_status;
    }

    /*-Set data format and start transfert-----------------------------------*/    
    if ((mxOX2000_status = viPrintf (instrSession,"TRAC? INT%hd\n",source)) < 0)
        return mxOX2000_status;

    if (interface==VI_INTF_ASRL) {
        /* Read 11 characters (#an header) to define the length of the binary block following */
        if ((mxOX2000_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
            return mxOX2000_status;

        headerSize = tabByte[1] - 0x30;
        Scan(tabByte,"%s[i2]>%i[b4]",&size);

        if ((mxOX2000_status = viRead (instrSession, tabByte + HEADER_SIZE, size+4+headerSize-HEADER_SIZE, &retCnt)) < 0)
             return mxOX2000_status;
        
        /*-Reset Protocol for binary file transfers----------------------------*/
        if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
            return mxOX2000_status;
    }
    else if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
        return mxOX2000_status;
    
    /*-Reset Time Out ---------------------------------------------------*/
    if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
        return mxOX2000_status;

    /*-Make data readable----------------------------------------------------*/
    if ((mxOX2000_status = mxOX2000_makeDataReadable(tabByte, interpolation,waveformArray,numberOfSamples,XStart,XIncrement)) < 0)
        return mxOX2000_status;
                            
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Read Channel To File                                            */
/* Purpose:  This function reads a trace from the instrument.                */
/*           Data is stored in a file                                        */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_readChToFile ( ViSession instrSession, ViInt16 channel,
                    ViInt16 drive, ViString filename, ViBoolean overwrite)
{                        
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViByte tstRes;
    ViInt16 headerSize;
    ViInt32 fileSize,retCnt;
    static ViByte tabByte[MAX_MEM_SIZE];
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(channel,1,4))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range(drive,0,2))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViBooleanRange(overwrite))
        return VI_ERROR_PARAMETER5;
        
    /*-Check if source is active---------------------------------------------*/
    if ((mxOX2000_status = viPrintf (instrSession,"INP%hd:STAT?\n",channel)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viScanf (instrSession,"%hd", &tstRes)) < 0)
        return mxOX2000_status;
    if (tstRes != 1)   
        return VI_ERROR_OX2000_CHAN_NOT_ACTIVE;
    
    if ((mxOX2000_status = viPrintf (instrSession,"INP%hd:SEL?\n",channel)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
        return mxOX2000_status;
    if (tabByte[0] != 'C')  
        return VI_ERROR_OX2000_CHAN_NOT_ACTIVE;
    
    /*-Check filename--------------------------------------------------------*/
    if (mxOX2000_invalidFilename (filename,drive,".TRC"))
        return VI_ERROR_PARAMETER4;
    
    /*-Store current trace into OX2000---------------------------------------*/
    if (drive) {
                            
        /*-Check overwrite---------------------------------------------------*/
        if ((mxOX2000_status = viPrintf (instrSession,"MMEM:CAT? \"%s\"\n", cmdDrive[drive])) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
            return mxOX2000_status;
        if (FindPattern (tabByte, 0, -1, filename, 0, 0) != -1) {
            if (overwrite) {
                if ((mxOX2000_status = viPrintf (instrSession,"MMEM:DEL \"%s\",\"%s\"\n", filename, cmdDrive[drive])) < 0)
                    return mxOX2000_status;
            }
            else {      
                return VI_ERROR_OX2000_EXISTING_FILE;
            }
        }
        
        /*-Perform transfer--------------------------------------------------*/
        if ((mxOX2000_status = viPrintf (instrSession,"MMEM:STOR:TRAC INT%hd,\"%s\",\"%s\"\n",
                            channel,filename ,cmdDrive[drive])) < 0)
            return mxOX2000_status;
    }
    
    /*-Store current trace into a PC file------------------------------------*/
    else {
            
        /*-Check overwrite---------------------------------------------------*/
        if (GetFileInfo(filename,&fileSize)  && !overwrite) 
            return VI_ERROR_OX2000_EXISTING_FILE;

        /*-Modify Time Out for big waveform transfers------------------------*/
        if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE,2 * TMO_VALUE)) < 0)
                return mxOX2000_status;

        /*-Modify Protocol for binary file transfers-------------------------*/
        viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
        if(interface==VI_INTF_ASRL){
            if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
                return mxOX2000_status;
        }

        /*-Perform transfert-------------------------------------------------*/    
        if ((mxOX2000_status = viPrintf (instrSession,"TRAC? INT%hd\n",channel)) < 0)
            return mxOX2000_status;

        if (interface==VI_INTF_ASRL) {
            /* Read 11 characters (#an header) to define the length of the binary block following */
            if ((mxOX2000_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
                return mxOX2000_status;

            headerSize = tabByte[1] - 0x30;
            Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);

            if ((mxOX2000_status = viRead (instrSession, tabByte+HEADER_SIZE, fileSize+4+headerSize-HEADER_SIZE, &retCnt)) < 0)
                 return mxOX2000_status;
        
            retCnt += HEADER_SIZE;
    
            /*-Reset Protocol for binary file transfers----------------------------*/
            if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
                return mxOX2000_status;
        }
        else if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
            return mxOX2000_status;
        
        /*-Reset Time Out ---------------------------------------------------*/
        if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE,TMO_VALUE)) < 0)
            return mxOX2000_status;

        /*-Fill PC file------------------------------------------------------*/    
        if ((ArrayToFile (filename, tabByte, VAL_UNSIGNED_CHAR, retCnt - 2,retCnt - 2, VAL_GROUPS_TOGETHER,
                VAL_GROUPS_AS_ROWS,VAL_CONST_WIDTH, 4, VAL_BINARY,VAL_TRUNCATE)) < 0) 
            return VI_ERROR_INSTR_FILE_WRITE;
    }

    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Read File To Array                                              */
/* Purpose:  This function reads a file from the PC or from the instrument   */
/*           Data is stored in an array                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_readFileToArr (ViSession instrSession, ViInt16 drive,
                    ViString filename, ViBoolean interpolation, 
                    ViReal64 _VI_FAR waveformArray[], ViPInt32 numberOfSamples, 
                    ViPReal64 XStart, ViPReal64 XIncrement)
{   
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViInt16 headerSize;
    ViInt32 retCnt,fileSize;
    static ViByte tabByte[MAX_MEM_SIZE];

    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(drive,0,2))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViBooleanRange(interpolation))
        return VI_ERROR_PARAMETER4;
        
    /*-Check filename--------------------------------------------------------*/
    if (mxOX2000_invalidFilename (filename,drive,".TRC"))
        return VI_ERROR_PARAMETER3;
        
    /*-Read file from OX2000-------------------------------------------------*/
    if (drive) {
    
        /*-Check filename----------------------------------------------------*/
        if ((mxOX2000_status = viPrintf (instrSession,"MMEM:CAT? \"%s\"\n",cmdDrive[drive])) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
            return mxOX2000_status;
        if (FindPattern (tabByte, 0, -1, filename, 0, 0) < 0)
            return VI_ERROR_OX2000_FILE_NOT_FOUND;
        
        /*-Modify Protocol for binary file transfers-------------------------*/
        viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
        if(interface==VI_INTF_ASRL){
            if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
                return mxOX2000_status;
        }

        /*-Perform transfer--------------------------------------------------*/
        if ((mxOX2000_status = viPrintf (instrSession,"MMEM:DATA? \"%s\",\"%s\"\n",filename,cmdDrive[drive])) < 0)
            return mxOX2000_status;

        if (interface==VI_INTF_ASRL) {
            /* Read 11 characters (#an header) to define the length of the binary block following */
            if ((mxOX2000_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
                return mxOX2000_status;

            headerSize = tabByte[1] - 0x30;
            Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);

            if ((mxOX2000_status = viRead (instrSession, tabByte+HEADER_SIZE, fileSize+4+headerSize-HEADER_SIZE, &retCnt)) < 0)
                 return mxOX2000_status;
        
            retCnt += HEADER_SIZE;
    
            /*-Reset Protocol for binary file transfers----------------------------*/
            if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
                return mxOX2000_status;
        }
        else if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
            return mxOX2000_status;
    }
    
    /*-Read file from PC-----------------------------------------------------*/
    else {
        /*-Check filename----------------------------------------------------*/
        if (!GetFileInfo(filename,&fileSize)) 
            return VI_ERROR_OX2000_FILE_NOT_FOUND;
            
        if ((FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, MAX_MEM_SIZE,MAX_MEM_SIZE,
                VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_BINARY)) < 0)
            return VI_ERROR_INSTR_FILE_OPEN;
    }
    
    /*-Make data readable----------------------------------------------------*/
    if ((mxOX2000_status = mxOX2000_makeDataReadable(tabByte,interpolation,waveformArray,numberOfSamples, XStart, XIncrement)) < 0)
        return mxOX2000_status;
    
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Write Array To Instrument Display Memory                        */
/* Purpose:  This function writes an array to the OX2000 display memory.     */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_writeArrToMem (ViSession instrSession, ViReal64 _VI_FAR waveformArray[],
                ViInt16 arraySize, ViInt16 voltSense,ViReal64 offset, ViInt16 timeBase, ViInt16 target)
{                                       
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViUInt32 retCnt;
    ViInt16 headerSize,inputState;
    static ViByte string[12000],tabByte[MAX_MEM_SIZE];

    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(arraySize,1,10000))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViInt16Range(voltSense,0,17))
        return VI_ERROR_PARAMETER5;
    if (mxOX2000_invalidViReal64Range(offset,-cmdOffsetMax[voltSense],
            cmdOffsetMax[voltSense]))
        return VI_ERROR_PARAMETER6;
    if (mxOX2000_invalidViInt16Range(timeBase,0,31))
        return VI_ERROR_PARAMETER7;
    if (mxOX2000_invalidViInt16Range(target,1,4))
        return VI_ERROR_PARAMETER8;
    
    /*-Force target to memory------------------------------------------------*/
    if ((mxOX2000_status = viPrintf (instrSession,"INP%hd 0\n",target)) < 0)
        return mxOX2000_status;

    /*-Encode data-----------------------------------------------------------*/
    if (mxOX2000_encodeArray (tabByte,waveformArray,arraySize,voltSense,offset,timeBase) < 0)
        return mxOX2000_status;
        
    /*-Perform transfer------------------------------------------------------*/
    headerSize = 4;
    if (arraySize > 8945)   
        headerSize = 5; 
    
    Fmt (string,"%s<TRAC INT%i[b2],#%i[b2]%i[b2]%s[t-w*]\n",target,headerSize,1054+arraySize,1054+arraySize,tabByte);
    if ((mxOX2000_status = viWrite (instrSession, string, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX2000_status;
    
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Write File To Instrument Display Memory                         */
/* Purpose:  This function fills the instrument display memory.              */
/*           Data are in the specified file.                                 */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_writeFileToMem (ViSession instrSession, ViInt16 drive,
                    ViString filename, ViInt16 memory)
{   
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViInt16 headerSize;
    ViUInt32 retCnt,fileSize;
    static ViByte string[12000],tabByte[MAX_MEM_SIZE];
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(drive,0,2))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range(memory,1,4))
        return VI_ERROR_PARAMETER4;
    
    /*-Check filename--------------------------------------------------------*/
    if (mxOX2000_invalidFilename (filename,drive,".TRC"))
        return VI_ERROR_PARAMETER3;
        
    /*-Write file from OX2000------------------------------------------------*/
    if (drive) {
        /*-Check filename----------------------------------------------------*/
        if ((mxOX2000_status = viPrintf (instrSession,"MMEM:CAT? \"%s\"\n",cmdDrive[drive])) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
            return mxOX2000_status;
        if (FindPattern (tabByte, 0, -1, filename , 0, 0) < 0)
            return VI_ERROR_OX2000_FILE_NOT_FOUND;
        
        /*-Perform transfer--------------------------------------------------*/
        if ((mxOX2000_status = viPrintf (instrSession,"MMEM:LOAD:TRACE MEM%hd,\"%s\",\"%s\"\n",
                    memory, filename,cmdDrive[drive])) < 0)
            return mxOX2000_status;
    }
    
    /*-Write file from PC----------------------------------------------------*/
    else {
        /*-Check filename----------------------------------------------------*/
        if (!GetFileInfo(filename,&fileSize)) 
            return VI_ERROR_OX2000_FILE_NOT_FOUND;
            
        /*-Read PC file------------------------------------------------------*/
        if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, 20, 20,VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_BINARY))
            return VI_ERROR_INSTR_FILE_OPEN;
        Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);
        headerSize = FindPattern (tabByte, 0, -1, "OX2", 0, 0);
        if (headerSize == -1)   
            return VI_ERROR_OX2000_INVALID_FILE_CONTENT;
        
        if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, headerSize + fileSize, headerSize + fileSize,
                     VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_BINARY))
            return VI_ERROR_INSTR_FILE_OPEN;
        
        /*-Perform transfer--------------------------------------------------*/
        Fmt(string,"%s<TRAC INT%i[b2],%s[t-w*q]\n",memory,headerSize + fileSize,tabByte);            
        
        if ((mxOX2000_status = viWrite (instrSession, string, NumFmtdBytes (),&retCnt)) < 0) 
            return mxOX2000_status;

    }

    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Write Array To Instrument File                                  */                                     
/* Purpose:  This function converts an array into an OX2000 file format.     */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_writeArrToFile (ViSession instrSession, ViReal64 _VI_FAR waveformArray[],
                    ViInt16 arraySize, ViInt16 voltSense, ViReal64 offset, ViInt16 timeBase,
                    ViInt16 drive, ViString filename,ViBoolean overwrite)
{                                       
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViInt16 headerSize;
    ViUInt32 retCnt,fileSize;
    ViBoolean tstRes;
    static ViByte string[12000],tabByte[MAX_MEM_SIZE];
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(arraySize,1,10000))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViInt16Range(voltSense,0,17))
        return VI_ERROR_PARAMETER4;
    if (mxOX2000_invalidViReal64Range(offset,-cmdOffsetMax[voltSense],
            cmdOffsetMax[voltSense]))
        return VI_ERROR_PARAMETER5;
    if (mxOX2000_invalidViInt16Range(timeBase,0,31))
        return VI_ERROR_PARAMETER6;
    if (mxOX2000_invalidViInt16Range(drive,0,2))
        return VI_ERROR_PARAMETER7;
    if (mxOX2000_invalidViBooleanRange(overwrite))
        return VI_ERROR_INSTR_PARAMETER9;
        
    /*-Check filename--------------------------------------------------------*/
    if (mxOX2000_invalidFilename (filename,drive,".TRC"))
        return VI_ERROR_PARAMETER7;
    
    /*-Store current trace into OX2000---------------------------------------*/
    if (drive) {
        /*-Check overwrite---------------------------------------------------*/
        if ((mxOX2000_status = viPrintf (instrSession,"MMEM:CAT? \"%s\"\n",cmdDrive[drive])) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
            return mxOX2000_status;
        if (FindPattern (tabByte, 0, -1, filename, 0, 0) != -1) {
            if (overwrite) {
                if ((mxOX2000_status = viPrintf (instrSession,"MMEM:DEL \"%s\",\"%s\"\n", filename, cmdDrive[drive])) < 0)
                    return mxOX2000_status;
            }
            else {      
                return VI_ERROR_OX2000_EXISTING_FILE;
            }
        }

        /*-Encode data-------------------------------------------------------*/
        if (mxOX2000_encodeArray (tabByte,waveformArray,arraySize,voltSense,offset,timeBase) < 0)
            return mxOX2000_status;
        
        /*-Perform transfer--------------------------------------------------*/
        headerSize = 4;
        if (arraySize > 8945) headerSize = 5; 
    
        Fmt (string,"%s<MMEM:DATA \"%s\",\"%s\",#%i[b2]%i[b2]%s[t-w*]\n",filename,cmdDrive[drive],
                    headerSize,1054+arraySize,1054+arraySize,tabByte);
        if ((mxOX2000_status = viWrite (instrSession, string, NumFmtdBytes (),&retCnt)) < 0) 
            return mxOX2000_status;
    }
    
    /*-Store current trace into a PC file------------------------------------*/
    else {
        /*-Check overwrite---------------------------------------------------*/
        if (GetFileInfo(filename,&fileSize)  && !overwrite) 
            return VI_ERROR_OX2000_EXISTING_FILE;
            
        /*-Encode data-------------------------------------------------------*/
        if (mxOX2000_encodeArray (tabByte,waveformArray,arraySize,voltSense,offset,timeBase) < 0)
            return mxOX2000_status;

        /*-Fill PC file------------------------------------------------------*/    
        headerSize = 4;
        if (arraySize > 8945) headerSize = 5; 
    
        Fmt (string,"%s<#%i[b2]%i[b2]%s[t-w*]\n",headerSize,1054+arraySize,1054+arraySize,tabByte);
        
        if ((ArrayToFile (filename, string, VAL_UNSIGNED_CHAR, NumFmtdBytes () - 1,NumFmtdBytes () - 1,
                    VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_CONST_WIDTH, 4, VAL_BINARY,VAL_TRUNCATE)) < 0)
            return VI_ERROR_INSTR_FILE_WRITE;
    }
    
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Save/Recall Setup From/To Instrument                            */
/* Purpose:  This function allows you to save the current setup of the       */
/*           instrument or to recall a setup stored in a file.               */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_saveSetup (ViSession instrSession, ViBoolean mode, ViInt16 drive,
                    ViString filename, ViBoolean overwrite)
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViInt16 headerSize;
    ViUInt32 retCnt,fileSize;
    static ViByte string[1000],tabByte[MAX_MEM_SIZE];
    ViByte buf[10];
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViBooleanRange(mode))
        return VI_ERROR_PARAMETER2;
    if (mxOX2000_invalidViInt16Range(drive,0,2))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViBooleanRange(overwrite))
        return VI_ERROR_PARAMETER5;
    
    /*-Check filename--------------------------------------------------------*/
    if (mxOX2000_invalidFilename (filename,drive,".CFG"))
        return VI_ERROR_PARAMETER4;
    
    /*-Save setup from OX2000------------------------------------------------*/
    if (mode) {
        /*-Target drive is OX2000--------------------------------------------*/
        if (drive) {
            /*-Check overwrite-----------------------------------------------*/
            if ((mxOX2000_status = viPrintf (instrSession,"MMEM:CAT? \"%s\"\n",cmdDrive[drive])) < 0)
                return mxOX2000_status;
            if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
                return mxOX2000_status;
            if (FindPattern (tabByte, 0, -1, filename, 0, 0) != -1) {
                if (overwrite) {
                    if ((mxOX2000_status = viPrintf (instrSession, "MMEM:DEL \"%s\",\"%s\"\n", filename, cmdDrive[drive])) < 0)
                        return mxOX2000_status;
                }
                else return VI_ERROR_OX2000_EXISTING_FILE;
            }
            /*-Perform transfer----------------------------------------------*/
            if ((mxOX2000_status = viPrintf (instrSession, "MMEM:STORE:STATE \"%s\",\"%s\"\n",filename,cmdDrive[drive])) < 0)
                return mxOX2000_status;
        }
        /*-Target drive is PC------------------------------------------------*/
        else {
            /*-Check overwrite-----------------------------------------------*/    
            if (GetFileInfo(filename,&fileSize)  && !overwrite) 
                return VI_ERROR_OX2000_EXISTING_FILE;
            
            /*-Modify Protocol for binary file transfers---------------------*/
            viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
            if(interface==VI_INTF_ASRL){
                if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
                    return mxOX2000_status;
            }
            /*-Perform transfer----------------------------------------------*/    
            if ((mxOX2000_status = viPrintf (instrSession,"SYST:SET?\n")) < 0)
                return mxOX2000_status;
            
            if (interface==VI_INTF_ASRL) {
                /* Read 11 characters (#an header) to define the length of the binary block following */
                if ((mxOX2000_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
                    return mxOX2000_status;

                headerSize = tabByte[1] - 0x30;
                Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);

                if ((mxOX2000_status = viRead (instrSession, tabByte + HEADER_SIZE, fileSize+4+headerSize-HEADER_SIZE, &retCnt)) < 0)
                     return mxOX2000_status;
                
                retCnt += HEADER_SIZE;
    
                /*-Reset Protocol for binary file transfers----------------------------*/
                if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
                    return mxOX2000_status;
            }
            else if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
                return mxOX2000_status;
            
            /*-Fill PC file--------------------------------------------------*/    
            if ((ArrayToFile (filename, tabByte, VAL_UNSIGNED_CHAR, retCnt - 2,retCnt - 2, VAL_GROUPS_TOGETHER,
                    VAL_GROUPS_AS_ROWS,VAL_CONST_WIDTH, 4, VAL_BINARY,VAL_TRUNCATE)) < 0) 
                return VI_ERROR_INSTR_FILE_WRITE;
        }
    }
    
    /*-Recall setup to OX2000------------------------------------------------*/
    else {
        /*-Source drive is OX2000--------------------------------------------*/
        if (drive) {
            /*-Check file name-----------------------------------------------*/    
            if ((mxOX2000_status = viPrintf (instrSession,"MMEM:CAT? \"%s\"\n",cmdDrive[drive])) < 0)
                return mxOX2000_status;
            if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE,&retCnt)) < 0)
                return mxOX2000_status;
            if (FindPattern (tabByte, 0, -1, filename, 0, 0) < 0)
                return VI_ERROR_OX2000_FILE_NOT_FOUND;
            
            /*-Perform transfer----------------------------------------------*/
            if ((mxOX2000_status = viPrintf (instrSession,"MMEM:LOAD:STATE \"%s\",\"%s\"\n",filename, cmdDrive[drive])) < 0)
                return mxOX2000_status;
        }
        /*-Source drive is PC------------------------------------------------*/
        else {
            /*-Check file name-----------------------------------------------*/    
            if (!GetFileInfo(filename,&fileSize)) 
                return VI_ERROR_OX2000_FILE_NOT_FOUND;
            /*-Read PC file--------------------------------------------------*/
            if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, 20, 20,VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
                return VI_ERROR_INSTR_FILE_OPEN;
            Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);
            headerSize = FindPattern (tabByte, 0, -1, "OX2", 0, 0);
            if (headerSize == -1)   
                return VI_ERROR_OX2000_INVALID_FILE_CONTENT;
            
            if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR,headerSize + fileSize, 
                    headerSize + fileSize, VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
                return VI_ERROR_INSTR_FILE_OPEN;
            /*-Perform transfer----------------------------------------------*/
            Fmt(string,"%s<SYST:SET %s[t-w*q]\n",headerSize + fileSize,tabByte);
            if ((mxOX2000_status = viWrite (instrSession, string, NumFmtdBytes (), &retCnt)) < 0) 
                return mxOX2000_status;
        }
    }
    
    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: File Transfer                                                   */ 
/* Purpose:  This function allows you to transfer a file from a drive        */
/*           to another.                                                     */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_fileTransfer (ViSession instrSession, ViString filename,
                    ViInt16 srcDrive, ViBoolean overwrite, ViInt16 tgtDrive)
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViByte trueFilename[MAX_FILENAME_LEN];
    ViInt16 headerSize;
    ViInt32 retCnt,retCnt1,retCnt2,fileSize;
    static ViByte tabByte[MAX_MEM_SIZE],string[MAX_MEM_SIZE],tempString[20];
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViInt16Range(srcDrive,0,2))
        return VI_ERROR_PARAMETER3;
    if (mxOX2000_invalidViBooleanRange(overwrite))
        return VI_ERROR_PARAMETER4;
    if (mxOX2000_invalidViInt16Range(tgtDrive,0,2))
        return VI_ERROR_PARAMETER5;
    
    /*-Check filename--------------------------------------------------------*/
    if ((FindPattern (filename, 0, -1, ".", 0, 0) != 8) && (StringLength(filename) != 12))
        return VI_ERROR_PARAMETER2;
    
    /*-Transfer--------------------------------------------------------------*/
    /*-No transfer if same drive---------------------------------------------*/
    if (srcDrive == tgtDrive) return mxOX2000_status;
    
    /*-OX2000 internal transfer----------------------------------------------*/
    if (srcDrive && tgtDrive) {
        /*-Check filename----------------------------------------------------*/
        if ((mxOX2000_status = viPrintf (instrSession,"MMEM:CAT? \"%s\"\n",cmdDrive[srcDrive])) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
            return mxOX2000_status;
        if (FindPattern (tabByte, 0, -1, filename, 0, 0) < 0)
            return VI_ERROR_OX2000_FILE_NOT_FOUND;
        /*-Check overwrite---------------------------------------------------*/    
        if ((mxOX2000_status = viPrintf (instrSession,"MMEM:CAT? \"%s\"\n",cmdDrive[tgtDrive])) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
            return mxOX2000_status;
        if (FindPattern (tabByte, 0, -1, filename, 0, 0) != -1) {
            if (overwrite) {
                if ((mxOX2000_status = viPrintf (instrSession,"MMEM:DEL \"%s\",\"%s\"\n", filename, cmdDrive[tgtDrive])) < 0)
                    return mxOX2000_status;
            }
            else   
                return VI_ERROR_OX2000_EXISTING_FILE;
        }
        
        /*-Perform transfer--------------------------------------------------*/
        if ((mxOX2000_status = viPrintf (instrSession,"MMEM:COPY \"%s\",\"%s\",\"%s\"\n",
                    filename, cmdDrive[srcDrive], cmdDrive[tgtDrive])) < 0)
            return mxOX2000_status;
    }
    
    /*-Computer to OX2000 transfer-------------------------------------------*/
    else if ((!srcDrive) && (tgtDrive > 0)) {
            
        /*-Check file name---------------------------------------------------*/    
        if (!GetFileInfo(filename,&fileSize)) 
            return VI_ERROR_OX2000_FILE_NOT_FOUND;
        SplitPath (filename, NULL, NULL, trueFilename);
            
        /*-Check overwrite---------------------------------------------------*/    
        if ((mxOX2000_status = viPrintf (instrSession,"MMEM:CAT? \"%s\"\n",cmdDrive[tgtDrive])) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
            return mxOX2000_status;
        if (FindPattern (tabByte, 0, -1, filename, 0, 0) != -1) {
            if (overwrite) {
                if ((mxOX2000_status = viPrintf (instrSession,"MMEM:DEL \"%s\",\"%s\"\n", filename, cmdDrive[tgtDrive])) < 0)
                    return mxOX2000_status;
            }
            else   
                return VI_ERROR_OX2000_EXISTING_FILE;
        }
        
        /*-Read PC file------------------------------------------------------*/
        StringUpperCase (trueFilename);
             
        /*-Read any file but ".TRC" or ".CFG"--------------------------------*/
        if ((FindPattern(trueFilename,0,-1,"TRC",0,0)==-1) && (FindPattern(trueFilename,0,-1,"CFG",0,0)==-1)) {
            GetFileInfo(filename,&fileSize);
            if (FileToArray (filename,tabByte,VAL_UNSIGNED_CHAR,fileSize,1,VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS,VAL_BINARY) < 0)
                return VI_ERROR_INSTR_FILE_OPEN;
            Fmt(tempString,"%s<%i[b4]",fileSize);
            
            /*-Modify Time Out for big file transfers------------------------*/
            if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, 5 * TMO_VALUE)) < 0)
                    return mxOX2000_status;

            /*-Perform transfer----------------------------------------------*/
            Fmt(string,"%s<MMEM:DATA \"%s\",\"%s\",#%i[b2]%i[b4]%s[t-w*q]\n\r",
                trueFilename,cmdDrive[tgtDrive],StringLength(tempString),fileSize,fileSize + 0L,tabByte);
            if ((mxOX2000_status = viWrite (instrSession, string, NumFmtdBytes (), &retCnt)) < 0) 
                return mxOX2000_status;
    
            /*-Reset Time Out -----------------------------------------------*/
            if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE,TMO_VALUE)) < 0)
                return mxOX2000_status;
        }
        
        /*-Read ".TRC" or ".CFG" files---------------------------------------*/
        else {
            if (FileToArray (filename,tabByte,VAL_UNSIGNED_CHAR,20,20,VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
                return VI_ERROR_INSTR_FILE_OPEN;
                
            Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);
            headerSize = FindPattern (tabByte, 0, -1, "OX2", 0, 0);
            
            if (headerSize == -1) 
                return VI_ERROR_OX2000_INVALID_FILE_CONTENT;
        
            if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, headerSize + fileSize, headerSize + fileSize,
                                VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
                return VI_ERROR_INSTR_FILE_OPEN;
                     
            /*-Perform transfer--------------------------------------------------*/
            Fmt(string,"%s<MMEM:DATA \"%s\",\"%s\",%s[t-w*q]\n\r",trueFilename,cmdDrive[tgtDrive],headerSize + fileSize,tabByte);
            if ((mxOX2000_status = viWrite (instrSession, string, NumFmtdBytes (), &retCnt)) < 0) 
                return mxOX2000_status;
        }
    }
    
    /*-OX2000 to Computer transfer-------------------------------------------*/
    else if ((srcDrive > 0) && (!tgtDrive)) {
    
        /*-Check filename----------------------------------------------------*/
        if ((mxOX2000_status = viPrintf (instrSession,"MMEM:CAT? \"%s\"\n",cmdDrive[srcDrive])) < 0)
            return mxOX2000_status;
        if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
            return mxOX2000_status;
        if (FindPattern (tabByte, 0, -1, filename, 0, 0) < 0)
            return VI_ERROR_OX2000_FILE_NOT_FOUND;
        
        /*-Check overwrite---------------------------------------------------*/    
        if (GetFileInfo(filename,&fileSize)  && !overwrite) 
            return VI_ERROR_OX2000_EXISTING_FILE;
            
        /*-Modify Time Out for big file transfers----------------------------*/
        if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE,25 * TMO_VALUE)) < 0)
            return mxOX2000_status;

        /*-Modify Protocol for binary file transfers-------------------------*/
        viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
        if(interface==VI_INTF_ASRL){
            if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
                return mxOX2000_status;
        }

        /*-Perform transfer--------------------------------------------------*/
        if ((mxOX2000_status = viPrintf (instrSession,"MMEM:DATA? \"%s\",\"%s\"\n",filename,cmdDrive[srcDrive])) < 0)
            return mxOX2000_status;

        if (interface==VI_INTF_ASRL) {
            /* Read 11 characters (#an header) to define the length of the binary block following */
            if ((mxOX2000_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
                return mxOX2000_status;

            headerSize = tabByte[1] - 0x30;
            Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);

            /* if file contains more 65536 bytes (color BMP file) -----------*/
            if (fileSize>65535) {
                if ((mxOX2000_status = viRead (instrSession,tabByte+HEADER_SIZE, 60000, &retCnt)) < 0)
                    return mxOX2000_status;
                if ((mxOX2000_status = viRead (instrSession,tabByte+HEADER_SIZE+60000, 60000, &retCnt1)) < 0)
                    return mxOX2000_status;
                if ((mxOX2000_status = viRead (instrSession,tabByte+HEADER_SIZE+120000,
                                        fileSize + 4 + headerSize - 120000 - HEADER_SIZE,&retCnt2))<0)
                    return mxOX2000_status;
                retCnt = retCnt + retCnt1 + retCnt2;
            }
            else if ((mxOX2000_status = viRead (instrSession, tabByte+HEADER_SIZE, fileSize+4+headerSize-HEADER_SIZE, &retCnt)) < 0)
                 return mxOX2000_status;
        
            retCnt += HEADER_SIZE;
    
            /*-Reset Protocol for binary file transfers----------------------------*/
            if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
                return mxOX2000_status;
        }
        else if ((mxOX2000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
            return mxOX2000_status;

        /*-Reset Time Out ---------------------------------------------------*/
        if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE,TMO_VALUE)) < 0)
            return mxOX2000_status;

        /*-Parse non ".TRC" or ".CFG" files----------------------------------*/
        if ((FindPattern(filename,0,-1,"TRC",0,0)==-1) && (FindPattern(filename,0,-1,"CFG",0,0)==-1))
            headerSize = tabByte[1] - 0x30 + 2; 

        /*-Fill PC file------------------------------------------------------*/    
        if ((ArrayToFile (filename, tabByte + headerSize, VAL_UNSIGNED_CHAR, retCnt - headerSize - 2, 1, 
                VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_CONST_WIDTH, 4, VAL_BINARY,VAL_TRUNCATE)) < 0) 
            return VI_ERROR_INSTR_FILE_WRITE;
    }

    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Write To Instrument                                             */
/* Purpose:  This function writes a command string to the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_writeInstrData (ViSession instrSession, ViString writeBuffer)
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    
    if ((mxOX2000_status = viPrintf (instrSession, "%s", writeBuffer)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Read Instrument Buffer                                          */
/* Purpose:  This function reads the output buffer of the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_readInstrData (ViSession instrSession, ViInt32 numBytes,
                    ViChar _VI_FAR rdBuf[], ViPInt32 bytesRead)
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    *bytesRead = 0L;
        
    if ((mxOX2000_status = viRead (instrSession, rdBuf, BUFFER_SIZE, bytesRead)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Reset                                                           */
/* Purpose:  This function resets the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_reset (ViSession instrSession)
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX2000_status = VI_SUCCESS;

    /*-Initialize the instrument to a known state----------------------------*/
    if ((mxOX2000_status = viWrite (instrSession, "\n*RST\n", 6, &retCnt)) < 0)
        return mxOX2000_status;

    if ((mxOX2000_status = mxOX2000_defaultInstrSetup (instrSession)) < 0)  
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Self-Test                                                       */
/* Purpose:  This function executes the instrument self-test and returns     */
/*           the result.                                                     */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_selfTest (ViSession instrSession, ViPInt16 testResult,
                    ViChar _VI_FAR testMessage[])
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViUInt32 retCnt = 0;
    ViByte rdBuffer[BUFFER_SIZE];
    
    if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE,20 * TMO_VALUE)) < 0)
        return mxOX2000_status; 
            
    if ((mxOX2000_status = viWrite (instrSession, "*TST?\n", 6, &retCnt)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viRead (instrSession, rdBuffer, 50, &retCnt)) < 0)
        return mxOX2000_status;
    if (Scan (rdBuffer, "%i[b1]", testResult) != 1)
        return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    
    if (*testResult) 
        Fmt(testMessage, "%s<Self test failed");
    else 
        Fmt(testMessage, "%s<No Error");  
    
    if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, 
            TMO_VALUE)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Error Query                                                     */
/* Purpose:  This function queries the instrument error queue, and returns   */
/*           the result. If the error query function is not supported by the */
/*           instrument, this function returns the warning                   */
/*           VI_WARN_NSUP_ERROR_QUERY.                                       */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_errorQuery (ViSession instrSession, ViPInt32 errCode,
                    ViChar _VI_FAR errMessage[])
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViByte rdBuffer[BUFFER_SIZE];
    
    if ((mxOX2000_status = viWrite (instrSession, "SYST:ERR?\n", 10, &retCnt)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        return mxOX2000_status;

    if (Scan (rdBuffer, "%i[b4],%s[t10]", errCode,errMessage) != 2)
        return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Error Message                                                   */
/* Purpose:  This function translates the error return value from the        */
/*           instrument driver into a user-readable string.                  */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_errorMessage (ViSession instrSession, ViStatus errorCode,
                    ViChar _VI_FAR errMessage[])
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViInt16 i;
    
    static mxOX2000_tStringValPair statusDescArray[] = {
        {VI_ERROR_PARAMETER1,   "ERROR: Parameter 1 out of range"},
        {VI_ERROR_PARAMETER2,   "ERROR: Parameter 2 out of range"},
        {VI_ERROR_PARAMETER3,   "ERROR: Parameter 3 out of range"},
        {VI_ERROR_PARAMETER4,   "ERROR: Parameter 4 out of range"},
        {VI_ERROR_PARAMETER5,   "ERROR: Parameter 5 out of range"},
        {VI_ERROR_PARAMETER6,   "ERROR: Parameter 6 out of range"},
        {VI_ERROR_PARAMETER7,   "ERROR: Parameter 7 out of range"},
        {VI_ERROR_PARAMETER8,   "ERROR: Parameter 8 out of range"},
        {VI_ERROR_INSTR_PARAMETER9,   "ERROR: Parameter 9 out of range"},
        {VI_ERROR_FAIL_ID_QUERY,"ERROR: Identification query failed"},
        {VI_ERROR_INSTR_FILE_OPEN,    "ERROR: Opening the specified file"},
        {VI_ERROR_INSTR_FILE_WRITE,   "ERROR: Writing to the specified file"},
        {VI_ERROR_INSTR_INTERPRETING_RESPONSE,           "ERROR: Interpreting the instrument's response"},
        {VI_ERROR_OX2000_ERROR,                  "ERROR: Instrument specific error"},
        {VI_ERROR_OX2000_INCONSISTENT_PARAMETER3,"ERROR: Parameter 3 is INCONSISTENT"},
        {VI_ERROR_OX2000_INCONSISTENT_PARAMETER5,"ERROR: Parameter 5 is INCONSISTENT"},
        {VI_ERROR_OX2000_INCONSISTENT_PARAMETER6,"ERROR: Parameter 6 is INCONSISTENT"},
        {VI_ERROR_OX2000_INCONSISTENT_PARAMETER7,"ERROR: Parameter 7 is INCONSISTENT"},
        {VI_ERROR_OX2000_AUTOTEST_PROBLEM,       "ERROR: Performing a self-test"}, 
        {VI_ERROR_OX2000_CHAN_NOT_ACTIVE,        "ERROR: Channel is not active"},
        {VI_ERROR_OX2000_NO_CHAN_ENABLED,        "ERROR: No channel is enabled"},
        {VI_ERROR_OX2000_FILE_NOT_FOUND,         "ERROR: File not available on support"},
        {VI_ERROR_OX2000_EXISTING_FILE,          "ERROR: Filename already used"},
        {VI_ERROR_OX2000_INVALID_FILE_CONTENT,   "ERROR: File content is invalid"},
        
        {VI_NULL, VI_NULL}  
    };
          
    mxOX2000_status = viStatusDesc (instrSession, errorCode, errMessage);
    if (mxOX2000_status == VI_WARN_UNKNOWN_STATUS) {
        for (i=0; statusDescArray[i].stringName; i++) {
            if (statusDescArray[i].stringVal == errorCode) {
                Fmt (errMessage, "%s<%s", statusDescArray[i].stringName);
                return (VI_SUCCESS);
            }
        }
        Fmt (errMessage, "%s<Unknown Error 0x%x[uw8p0]", errorCode);
        return (VI_WARN_UNKNOWN_STATUS);
    }
    
    mxOX2000_status = VI_SUCCESS;
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Revision Query                                                  */
/* Purpose:  This function returns the driver and instrument revisions.      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_revisionQuery (ViSession instrSession,
                    ViChar _VI_FAR driverRev[], ViChar _VI_FAR instrRev[])
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViByte rdBuffer[BUFFER_SIZE];

    if ((mxOX2000_status = viWrite (instrSession, "*IDN?\n", 6, &retCnt)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        return mxOX2000_status;

    if (Scan (rdBuffer, "%s[t10]", instrRev) != 1)
        return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    Fmt (driverRev, "%s<%s", OX2000_REVISION);
    
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Front Panel Lockout                                             */
/* Purpose:  This function locks the instrument front panel.                 */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_frontPanLock (ViSession instrSession, ViBoolean mode)
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    
    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX2000_invalidViBooleanRange(mode))
        return VI_ERROR_PARAMETER2;
        
    if ((mxOX2000_status = viPrintf (instrSession, "SYST:KLOCK %hd\n", mode)) < 0)
        return mxOX2000_status;

    /*-Check instrument status & Return from function------------------------*/
    if ((mxOX2000_status = mxOX2000_instrStatus(instrSession)) < 0)
        return mxOX2000_status;
        
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Close                                                           */
/* Purpose:  This function closes the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX2000_close (ViSession instrSession)
{
    mxOX2000_instrRange instrPtr;
    ViSession rmSession;
    ViStatus mxOX2000_status = VI_SUCCESS;

    if ((mxOX2000_status = viGetAttribute (instrSession, VI_ATTR_RM_SESSION, &rmSession)) < 0)
        return mxOX2000_status;
    if ((mxOX2000_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr)) < 0)
        return mxOX2000_status;
    
    if(instrPtr != NULL)
        free (instrPtr);
    
    mxOX2000_status = viClose (instrSession);
    viClose (rmSession);

    return mxOX2000_status;
}

/*****************************************************************************/
/*= UTILITY ROUTINES (Non-Exportable Functions) =============================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Make data readable                                              */
/* Purpose:  This function fills output parameters with readable data        */
/*===========================================================================*/
ViStatus mxOX2000_makeDataReadable (ViByte tabByte[],ViBoolean interpolation,
                    ViReal64 _VI_FAR waveformArray[], ViPInt32 numberOfSamples,
                    ViPReal64 XStart, ViPReal64 XIncrement)
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViPByte buffer = (ViPByte) ptrTrace;
    ViByte start;
    ViInt16 i, j = 0, firstIndex, startIndex, stopIndex;
    ViReal64 slope;
    
    /*-Find begining of data file--------------------------------------------*/    
    start = FindPattern (tabByte, 0, 30, "OX2", 1, 0);
    if (start == -1) 
        return VI_ERROR_OX2000_INVALID_FILE_CONTENT;
    
    /*-Fill OX2000Trace structure with correct values------------------------*/    
    for (i=0;i<16;i++) {
        *(buffer + i) = tabByte[start + i];
    }
    
    for (i=16;i<32;i++) {
        if (i % 2)  
            *(buffer + i - 1) = tabByte[start + i];
        else        
            *(buffer + i + 1) = tabByte[start + i];
    }
    
    *(buffer + 35) = tabByte[start + 32];
    *(buffer + 34) = tabByte[start + 33];
    *(buffer + 33) = tabByte[start + 34];
    *(buffer + 32) = tabByte[start + 35];

    for (i=36;i<50;i++) {
        if (i % 2)  
            *(buffer + i - 1) = tabByte[start + i];
        else        
            *(buffer + i + 1) = tabByte[start + i];
    }
    
    *(buffer + 50) = tabByte[start + 50];
    *(buffer + 51) = tabByte[start + 51];
    
    for (i=52;i<1054;i++) {
        if (i % 2)  
            *(buffer + i - 1) = tabByte[start + i];
        else        
            *(buffer + i + 1) = tabByte[start + i];
    }
    
    
    /*-Fill traceInfo structure with readable data----------------------------*/    
    traceInfo.probeValue = pow(10,trace.probe);
    traceInfo.offsetValue = - trace.offsetHard * trace.mantisseVolt * pow(10,trace.expVolt - 15);
    traceInfo.timeBaseValue = cmdTimeBase[trace.timeBase] * trace.timeVernier / 128.0;
    traceInfo.voltSenseValue = cmdVolt[trace.voltSense] * trace.voltVernier / 80.0 * traceInfo.probeValue;
    traceInfo.sizeValue = cmdSize[trace.trigAndSize & 0x0F];
    traceInfo.pretrigValue = 10 * (trace.trigAndSize >> 4);
    traceInfo.couplingValue = trace.coupling;
    traceInfo.bandwidthValue = trace.bandwidth;
    
    /*-Fill output parameters with readable data-----------------------------*/    
    for (i=0;i<trace.sample;i++) {
        if (trace.vernier[i%trace.storageIndex] != INVALID_SAMPLE) {
            waveformArray[i] = ((tabByte[start + 1054 + i] * 399.0/255.0 - 200)* 
                                trace.mantisseVolt * pow(10,trace.expVolt - 15)) * 
                                (1 - 2 * trace.invertState) - traceInfo.offsetValue;
            j++;
        }
        else 
            waveformArray[i] = INVALID_VALUE;
    }
    *XIncrement = .01 * traceInfo.timeBaseValue;
    *XStart = 64 / (ViReal64) trace.timeVernier * (trace.runDelay - traceInfo.sizeValue * 0.01 * traceInfo.pretrigValue) 
                    * trace.mantisseTime * pow(10,trace.expTime - 15);
    *numberOfSamples = j;
    
    /*-Interpolate on request------------------------------------------------*/    
    if ((j != trace.sample) && interpolation) {
        i = 0;
        while (waveformArray[i] > 0.1 * INVALID_VALUE) i++;
        for (j=0;j<i;j++) {
            waveformArray[j] = waveformArray[i];
        }
        do {
            while (waveformArray[i] > 0.1 * INVALID_VALUE) i++;
            startIndex = i++;
            while (waveformArray[i] > 0.1 * INVALID_VALUE) i++;
            stopIndex = i;
            slope = (waveformArray[stopIndex] - waveformArray[startIndex]) / (ViReal64) (stopIndex - startIndex);
            for (i=startIndex;i<stopIndex;i++) {
                waveformArray[i] = slope * (i-startIndex) + waveformArray[startIndex];
            }
        } while (i <= (trace.sample - (stopIndex - startIndex)));
        for (i=stopIndex;i<trace.sample;i++) {
            waveformArray[i] = waveformArray[stopIndex];
        }
        
    }

    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Fill OX2000 trace structure                                     */
/* Purpose:  This function encodes an array into an OX2000 trace structure   */
/*===========================================================================*/
ViStatus mxOX2000_encodeArray (ViByte tabByte[],ViReal64 _VI_FAR waveformArray[],
                    ViInt16 arraySize, ViInt16 voltSense,
                    ViReal64 offset, ViInt16 timeBase)
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViInt16 i;
    ViInt32 sampleValue;
    
    /*-Set title-------------------------------------------------------------*/
    Fmt (tabByte,"%s<OX2000_TRC");
    
    /*-Set offset value------------------------------------------------------*/
    tabByte[16] = (-RoundRealToNearestInteger(50 * offset / cmdVoltSense[voltSense])) >> 8;
    tabByte[17] = (-RoundRealToNearestInteger(50 * offset / cmdVoltSense[voltSense])) & 0x00FF;
    tabByte[18] = tabByte[16];
    tabByte[19] = tabByte[17];
    
    /*-Set time base / number of samples / DC coupling-----------------------*/
    tabByte[23] = timeBase + 1;
    tabByte[25] = 128;
    tabByte[26] = arraySize >> 8;        
    tabByte[27] = arraySize; 
    tabByte[37] = 1;            

    /*-Set time mantissa and exponent----------------------------------------*/
    tabByte[42] = cmdTimeMantissa[timeBase%3] >> 8;          
    tabByte[43] = cmdTimeMantissa[timeBase%3];               
    tabByte[45] = cmdTimeExponent[timeBase];                 
    
    /*-Set voltage mantissa and exponent-------------------------------------*/
    tabByte[46] = cmdVoltMantissa[voltSense%3] >> 8;             
    tabByte[47] = cmdVoltMantissa[voltSense%3];              
    tabByte[49] = cmdVoltExponent[voltSense];                
    
    /*-Set dot-join display mode and storage index---------------------------*/
    tabByte[51] = 1;                 
    tabByte[53] = 2;                 


    /*-Calculate sample values [0 to 255]------------------------------------*/
    for (i=0;i<arraySize;i++) {
        sampleValue = RoundRealToNearestInteger( (200 + (waveformArray[i] + offset) /
                (cmdVoltMantissa[voltSense%3] * 
                pow(10,cmdVoltExponent[voltSense] - 15)) ) * 255.0 / 399.0 ) ;
        tabByte[1054+i] = sampleValue;
        if (sampleValue  < 0) tabByte[1054+i] = 0;
        if (sampleValue  > 255) tabByte[1054+i] = 255;
        
    }
    
    /*-Set voltage sense and probe value-------------------------------------*/
    if (voltSense > 11) {
        voltSense -= 6;
        tabByte[39] = 2;
    }
    tabByte[29] = voltSense;    
    tabByte[31] = 80;                  

    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Check File Name                                                 */
/* Purpose:  This function checks if the file name is properly edited        */
/*===========================================================================*/
ViBoolean mxOX2000_invalidFilename (ViString filename, ViInt16 drive, ViString type)
{
    ViInt16 startIndex;
    
    if (!drive) {
        startIndex = FindPattern (filename, 0, -1, "\\", 0, 1) + 1;
        if (startIndex < 0)  
            startIndex = 0;
        if (StringLength(filename + startIndex) != 12) 
            return VI_TRUE;
        StringUpperCase (filename);
        if (CompareStrings (filename, startIndex + 8, type, 0, 0) != 0)  
            return VI_TRUE; 
    }
    else {
        if (StringLength(filename) != 12)  
            return VI_TRUE;
        StringUpperCase (filename);
        if (CompareStrings (filename, 8, type, 0, 0) != 0)  
            return VI_TRUE; 
    }
        
    return VI_FALSE;
}

/*===========================================================================*/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*===========================================================================*/
ViBoolean mxOX2000_invalidViBooleanRange (ViBoolean val)
{
    return ((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Signed Integer Value Out Of Range - ViInt16               */
/* Purpose:  This function checks a short signed integer value to see if it  */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean mxOX2000_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean mxOX2000_invalidViInt32Range  (ViInt32 val, ViInt32 min, ViInt32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Unsigned Integer Value Out Of Range - ViUInt16            */
/* Purpose:  This function checks a short unsigned integer value to see if it*/  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean mxOX2000_invalidViUInt16Range  (ViUInt16 val, ViUInt16 min, ViUInt16 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Unsigned Integer Value Out Of Range - ViUInt32             */
/* Purpose:  This function checks a long unsigned integer value to see if it */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean mxOX2000_invalidViUInt32Range  (ViUInt32 val, ViUInt32 min, ViUInt32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Float) Value Out Of Range - ViReal32                      */
/* Purpose:  This function checks a real (float) value to see if it lies     */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean mxOX2000_invalidViReal32Range  (ViReal32 val, ViReal32 min, ViReal32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (double) value to see if it lies    */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean mxOX2000_invalidViReal64Range  (ViReal64 val, ViReal64 min, ViReal64 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Instrument Status                                               */
/* Purpose:  This function checks the instrument error status, by reading    */
/*           and processing instrument's event status register. An error     */
/*           is reported if bits 4 and 5 resp. EXE and CME are asserted.     */
/*===========================================================================*/
ViStatus mxOX2000_instrStatus (ViSession instrSession)
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViByte rdBuffer[BUFFER_SIZE];
    ViByte status;
    
    /*- Command to query Event Status Register ------------------------------*/
    if ((mxOX2000_status = viPrintf (instrSession,"*ESR?\n")) < 0)
        return mxOX2000_status;
        
    if ((mxOX2000_status = viRead (instrSession, rdBuffer, 50, &retCnt)) < 0)
        return mxOX2000_status;

    /*- Scanning the response -----------------------------------------------*/
    if (Scan (rdBuffer, "%i[b1]", &status) != 1)
        return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    
    /*- Testing bits 4 and 5 ------------------------------------------------*/
    if (status & 0x30) 
        return VI_ERROR_OX2000_ERROR;
    
    return mxOX2000_status;
}

/*===========================================================================*/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the mxOX2000_init function.  When */
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from mxOX2000_init to     */
/*           VI_NULL.                                                        */
/*===========================================================================*/
ViStatus mxOX2000_initCleanUp (ViSession openRMSession,ViPSession openInstrSession,
                    ViStatus currentStatus)
{
    viClose (*openInstrSession);
    viClose (openRMSession);
    *openInstrSession = VI_NULL;
    
    return currentStatus;
}

/*===========================================================================*/
/* Function: Default Instrument Setup                                        */
/* Purpose:  This function sends a default setup to the instrument.  This    */
/*           function is called by the mxOX2000_reset operation and by the   */
/*           mxOX2000_init function if the reset option has not been         */
/*           selected.  This function is useful for configuring any          */
/*           instrument settings that are required by the rest of the        */
/*           instrument driver functions such as turning headers ON or OFF   */
/*           or using the long or short form for commands, queries, and data.*/                                    
/*===========================================================================*/
ViStatus mxOX2000_defaultInstrSetup (ViSession instrSession)
{
    ViStatus mxOX2000_status = VI_SUCCESS;
    ViUInt32 retCnt = 0;
    mxOX2000_instrRange instrPtr;
        
    /* Determine if the structure has been initialized for the current VISA  */
    /* Session and malloc if it has not.                                     */
    if (mxOX2000_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
        return mxOX2000_status;
    
    if (instrPtr == NULL) 
        instrPtr = malloc (sizeof (struct mxOX2000_statusDataRanges));
    
    instrPtr -> triggerMode = 0;
    instrPtr -> val2 = 0;
    instrPtr -> val3 = 0;
    Fmt (instrPtr -> instrDriverRevision, "%s<%s", OX2000_REVISION);
    
    if ((mxOX2000_status = viSetAttribute (instrSession, VI_ATTR_USER_DATA, (ViUInt32)instrPtr)) < 0)
        return mxOX2000_status;                                       

    if ((mxOX2000_status = viWrite (instrSession, "*CLS;:FORM INT;:FORM:DINT OFF\n", 30, &retCnt)) < 0)
        return mxOX2000_status;

    return mxOX2000_status;    
}

/*****************************************************************************/
/*=== END INSTRUMENT DRIVER SOURCE CODE =====================================*/
/*****************************************************************************/
