#include "mtx3354.h"
#include <userint.h>
#include <utility.h>
#include <formatio.h>
#include <visa.h>		  
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/*==================================================================================*/
#define MTX3354_REVISION   "Rev 1.0, 07/2007,CVI 7.0"/* Instrument driver revision  */
#define BUFFER_SIZE        512L                      /* File I/O buffer size        */
#define INVALID_SAMPLE     0xFFFF                    /* Sample validity criterion   */
#define MAX_MEM_SIZE       200300                    /* Memory needed for file I/O  */
#define INVALID_VALUE      1E100                     /* Invalid sample default value*/
#define TMO_VALUE          4000                      /* Time out value              */
#define TMO_TRANSFER       20000                     /* Time out value              */
#define HEADER_SIZE        200                       /* Size of binary data header  */
#define HEADER_FILE        20                        /* Size of binary file header  */
/*==================================================================================*/

/*=MTX3354===================================================================*/
/* LabWindows/CVI 7.1.1 Instrument Driver (VISA I/O)                         */
/* Original Release: April 1998                                              */
/*                                                                           */                           
/* Metrix Technical Support: Olivier Lemaire                                 */
/*                                                                           */
/* For LabWindows Technical Support in the United States call:               */
/*              National Instruments, Austin, Texas                          */
/*              Ph (800) 433-3488    Fax (512) 794-5678                      */
/*                                                                           */
/*                                                                           */
/* Modification History: None                                                */
/*                                                                           */
/*===========================================================================*/
 
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS =====================================*/
/*****************************************************************************/
/*-Vertical command arrays---------------------------------------------------*/
static ViString cmdVertCoupling[] = {"AC","DC","GRO"};       
static ViString cmdVoltDiv[] = {"20mV","40mV","80mV", "160mV","400mV","800mV",
						"1.6V","4V","8V", "16V","40V","80V", "160V","400V","800V",
						"1.6kV","4kV","8kV"}; 
static ViReal64 cmdOffsetMax[] = {80E-3,80E-3,400E-3,400E-3,2.0,2.0,2.5,12.5,12.5,
						64.0,64.0,64.0,640.0,640.0,640.0,6400.0,6400.0,6400.0}; 

/*-Horizontal command arrays-------------------------------------------------*/
static ViString cmdTimeDiv[] = {"1E-9", "2E-9", "5E-9", "10E-9", "25E-9", "50E-9",
								"125E-9", "250E-9", "500E-9", "1E-6", "2.5E-6", "5E-6", "10E-6", "25E-6",
						"50E-6", "100E-6", "200E-6", "500E-6", "1E-3", "2E-3", "5E-3",
						"10E-3", "20E-3", "50E-3", "100E-3", "200E-3", "500E-3","1",
						"2" ,"5", "10", "20", "50", "100", "200"};
                        
static ViReal64 cmdOffset[] = {1E-9, 1E-9, 1E-9, 1E-9, 1E-9, 1E-9, 1E-9, 1E-9, 1E-9,
							1E-6, 1E-6, 1E-6, 1E-6, 1E-6, 1E-6, 1E-6, 1E-6, 1E-6,
							1E-3, 1E-3, 1E-3, 1E-3, 1E-3, 1E-3, 1E-3, 1E-3, 1E-3,
							1,    1,    1,    1,    1,    1,    1,    1};
static ViString cmdDispTrace[] = {"A", "XY"}; 
static ViString cmdcalcfft[] = {"RECT", "HAMM", "HANN", "BLAC"};
static ViString cmdScale[] = {"LIN", "LOG"};

/*-Trigger command arrays---------------------------------------------------*/
static ViString cmdTrigCoupling[] = {"AC","DC","ON"};       
static ViString cmdSlope[] = {"NEG","POS"};
static ViString cmdSource[]= {"INT1", "INT2", "INT3", "INT4", "LINE"}; 
static ViString cmdStandVid[] = {"525", "625"};
static ViString cmdSlopeVid[] = {"POS", "NEG"};   
static ViString CmdPulsTyp[] = {"EQU", "SUP", "INF"};

/*-Display command arrays----------------------------------------------------*/
static ViString cmddevice[]={"SCOP", "ANALYS", "REC","SCOP"};
static ViString cmdDisplayLang[] = {"ENG", "FRE","GER", "SPA", "ITA"};
static ViString cmdDisplayScreen[] = {"0", "15", "30", "60"};
static ViString cmdSystemStandby[] = {"30", "60", "240", "1440"};
static ViString cmdMaxminAcq[] = {"ENV", "NORM"};
static ViString cmdMode[] = {"NORM", "ENV"};       
static ViString cmdAverage[] = {"0", "2", "4", "16", "64"};       
static ViString cmdPalette[] = {"MONO", "MULTI"};
static ViString cmdTiming[] = {"100ms", "200ms", "500ms", "1s", "2s", "5s", "10s", "INF"};

/*-Hardcopy command arrays---------------------------------------------------*/
static ViString cmdDestination[] = {"SER", "CENT", "NET", ""};       
static ViString cmdHardcopyLang[] = {"IBM", "EPS", "CAN", "PCL", "DPU", "POST", "BMP", "GIF"}; 
/*-Read command arrays-------------------------------------------------------*/
static ViString cmdReadMeasure[] = {"MIN", "MAX", "PTP", "LOW", "HIGH", "AMPL", "AC","VOLT",
							"RISE:OVER", "FALL:OVER", "RTIME", "FTIME", "PWID","NWID", "PER", "FREQ", "PDUT",
							"PUL:COUN", "PHAS"};  
static ViReal64 cmdTimeBase[] = {1E-9, 2E-9, 5E-9, 10E-9, 20E-9, 50E-9, 100E-9, 200E-9, 500E-9,
							1E-6, 2E-6, 5E-6, 10E-6, 20E-6, 50E-6, 100E-6, 200E-6, 500E-6, 1E-3,
							2E-3, 5E-3, 10E-3, 20E-3, 50E-3, 100E-3, 200E-3, 500E-3, 1.0, 2.0, 5.0,
							10.0, 20.0, 50.0, 100.0};
static ViReal64 cmdVolt[] = {2.5E-3,5E-3,10E-3,20E-3,50E-3,100E-3,200E-3,500E-3,1.0,
							2.0,5.0,10.0,20.0,50.0,100.0,200.0};
static ViString cmdRef[] = {"INT1", "INT2"};
                                     
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT STATUS/RANGE STRUCTURE  ============================*/
/*****************************************************************************/
/*  mtx3354_stringValPair is used in the  mtx3354_errorMessage function      */
/*===========================================================================*/
typedef struct   mtx3354_stringValPair
{
   ViStatus stringVal;
   ViPString stringName;
}   mtx3354_tStringValPair;
	 
struct mtx3354_statusDataRanges {
    ViInt16 triggerMode;
    ViInt16 val2;
    ViInt16 val3;
    ViChar instrDriverRevision[256];
    ViInt16 instrMode;
};
typedef struct mtx3354_statusDataRanges * mtx3354_instrRange;

ViInt16 TrigMode = 0;

/*****************************************************************************/
/*= UTILITY ROUTINE DECLARATIONS (Non-Exportable Functions) =================*/
/*****************************************************************************/
ViBoolean mtx3354_invalidViBooleanRange (ViBoolean val);
ViBoolean mtx3354_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max);
ViBoolean mtx3354_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max);
ViBoolean mtx3354_invalidViUInt16Range (ViUInt16 val, ViUInt16 min, ViUInt16 max);
ViBoolean mtx3354_invalidViUInt32Range (ViUInt32 val, ViUInt32 min, ViUInt32 max);
ViBoolean mtx3354_invalidViReal32Range (ViReal32 val, ViReal32 min, ViReal32 max);
ViBoolean mtx3354_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max);
ViStatus  mtx3354_makeDataReadable (ViByte tabByte[], ViReal64 _VI_FAR waveformArray[], ViPInt32 numberOfSamples,
                                ViPReal64 XStart, ViPReal64 XIncrement);
ViBoolean mtx3354_invalidFilename (ViString filename, ViInt16 drive, ViString type);
ViStatus  mtx3354_instrStatus (ViSession instrSession);
ViStatus  mtx3354_initCleanUp (ViSession openRMSession, ViPSession openInstrSession, ViStatus currentStatus);
ViStatus  mtx3354_defaultInstrSetup (ViSession openInstrSession);

/*****************************************************************************/
/*====== USER-CALLABLE FUNCTIONS (Exportable Functions) =====================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function:	Initialize                                                   */
/* Purpose:		This function opens the instrument, queries the instrument   */
/*				for its ID, and initializes the instrument to a known state  */
/*===========================================================================*/
ViStatus _VI_FUNC  mtx3354_init (ViRsrc resourceName, ViBoolean IDQuery,
					ViPSession instrSession)
{
	ViStatus  mtx3354_status = VI_SUCCESS;
	ViSession rmSession = 0;
	ViUInt32 retCnt = 0;
	ViByte rdBuffer[BUFFER_SIZE], buff[10];
	ViUInt16 interface, i;
    
	/*- Check input parameter ranges ----------------------------------------*/
	if ( mtx3354_invalidViBooleanRange (IDQuery))
		return VI_ERROR_PARAMETER2;

	/*- Open instrument session ---------------------------------------------*/

	if (( mtx3354_status = viOpenDefaultRM (&rmSession)) < 0)
		return  mtx3354_status;

	if (FindPattern (resourceName, 0, 3, "USB", 0, 0) != -1)
		{
		i = 2;
		while (i < 20)
	    	{
	    	Fmt (buff, "ASRL%i", i);
		    if (viOpen (rmSession, buff, VI_NULL, VI_NULL, instrSession) == 0)
				{
			    viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, 200);
			    viSetBuf (*instrSession, VI_READ_BUF|VI_WRITE_BUF, 50000);
				viSetAttribute (*instrSession, VI_ATTR_WR_BUF_OPER_MODE, VI_FLUSH_ON_ACCESS);
			    viSetAttribute (*instrSession, VI_ATTR_RD_BUF_OPER_MODE, VI_FLUSH_ON_ACCESS);
				viSetAttribute (*instrSession, VI_ATTR_ASRL_FLOW_CNTRL, VI_ASRL_FLOW_RTS_CTS);
				viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, 0x0D);
				viSetAttribute (*instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR);
				viSetAttribute (*instrSession, VI_ATTR_ASRL_BAUD, 921600);
				viWrite (*instrSession, "\r*IDN?\r", 7, &retCnt);
				viRead (*instrSession, rdBuffer, 50, &retCnt);
				if (retCnt != 0)
					{
					if (rdBuffer[0] == 0)
						rdBuffer[0] = 0x20;

					if (FindPattern (rdBuffer, 0, -1, "MTX3354", 0, 0) != -1)
						break;
					}

		        viClose (*instrSession);
				}
			i ++;
			}
		if (i >= 20)
			{
	        viClose (rmSession);
	        mtx3354_status = 0xBFFF0015;
			return mtx3354_status;
			}
		}
	else
		{
		if (( mtx3354_status = viOpen (rmSession, resourceName, VI_NULL, VI_NULL, instrSession)) < 0) 
			{
			viClose (rmSession);
			return  mtx3354_status;
			}	

		viGetAttribute (*instrSession, VI_ATTR_INTF_TYPE, &interface);

		/* Serial Interface initialized to 115200 bauds, 8 databits, 1 stopbit, no parity, RTS/CTS protocol*/
		if(interface==VI_INTF_ASRL) 
			{
			if (( mtx3354_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_BAUD, 115200)) < 0)
				return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
			if (( mtx3354_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_DATA_BITS, 8)) < 0)
				return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
			if (( mtx3354_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_STOP_BITS, VI_ASRL_STOP_ONE)) < 0)
				return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
			if (( mtx3354_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_PARITY, VI_ASRL_PAR_NONE)) < 0)
				return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
			if (( mtx3354_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_FLOW_CNTRL, VI_ASRL_FLOW_RTS_CTS)) < 0)
				return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
			if (( mtx3354_status = viSetBuf (*instrSession, VI_ASRL_IN_BUF|VI_ASRL_OUT_BUF, 5000)) < 0)
				return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
			if (( mtx3354_status = viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, 0x0D)) < 0)
				return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
			if (( mtx3354_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
				return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
			if (( mtx3354_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_END_OUT, VI_ASRL_END_NONE)) < 0)
				return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
			if (( mtx3354_status = viWrite (*instrSession, "\r", 1, &retCnt)) < 0)
				return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
			}
		}

	/*- Configure VISA Formatted I/O ----------------------------------------*/
	if (( mtx3354_status = viSetAttribute (*instrSession, VI_ATTR_TERMCHAR_EN, VI_FALSE)) < 0)
		return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
	if (( mtx3354_status = viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
		return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
	if (( mtx3354_status = viSetBuf (*instrSession, VI_READ_BUF|VI_WRITE_BUF, 4000)) < 0)
		return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
	if (( mtx3354_status = viSetAttribute (*instrSession, VI_ATTR_WR_BUF_OPER_MODE,VI_FLUSH_ON_ACCESS)) < 0)
		return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
	if (( mtx3354_status = viSetAttribute (*instrSession, VI_ATTR_RD_BUF_OPER_MODE,VI_FLUSH_ON_ACCESS)) < 0)
		return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);

	/*- Identification Query ------------------------------------------------*/
	if (IDQuery) 
		{
		if (( mtx3354_status = viWrite (*instrSession, "*IDN?\r", 6, &retCnt)) < 0)
			return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
		if (( mtx3354_status = viRead (*instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
			return  mtx3354_status;
		if (FindPattern (rdBuffer, 0, -1, "MTX3354", 1, 0) == -1)
			return  mtx3354_initCleanUp (rmSession, instrSession, VI_ERROR_FAIL_ID_QUERY);
		}

	/*- Send Default Instrument Setup ---------------------------------*/
																			   
	if (( mtx3354_status =  mtx3354_defaultInstrSetup (*instrSession)) < 0)
		return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
	
	if (( mtx3354_status = viWrite (*instrSession, "FORM INT;FORM:DINT 1\r", 21, &retCnt)) < 0)
		return  mtx3354_initCleanUp (rmSession, instrSession,  mtx3354_status);
	
	/*-Check instrument status & Return from function------------------------*/
	if (( mtx3354_status =  mtx3354_instrStatus(*instrSession)) < 0)
		return  mtx3354_status;

	return  mtx3354_status;
}

/*===========================================================================*/
/* Function: Example                                                         */
/* Purpose:  This function is an example of this driver's functions use.     */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_example (ViSession instrSession, ViInt16 channel,
					ViInt16 timebase, ViInt16 voltageRange, ViInt16 coupling, 
					ViReal64 _VI_FAR data[], ViPInt32 numberOfSamples, 
					ViPReal64 XStart, ViPReal64 XIncrement)
{
	ViStatus mtx3354_status = VI_SUCCESS;
    
	/*- Check input parameter ranges ----------------------------------------*/    
	if (mtx3354_invalidViInt16Range (channel, 0, 3))
		return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViInt16Range (timebase, 0, 33))
		return VI_ERROR_PARAMETER3;
	if (mtx3354_invalidViInt16Range (voltageRange, 0, 17))
		return VI_ERROR_PARAMETER4;
	if (mtx3354_invalidViInt16Range (coupling, 0, 2))
		return VI_ERROR_PARAMETER5;
    
	/*- Configuring vertical settings----------------------------------------*/
	if((mtx3354_status = mtx3354_verticalSetup (instrSession, channel + 1, VI_ON, coupling, 1, voltageRange, 0.0, 0)) < 0)
		return mtx3354_status;
        
	/*- Setting timebase ----------------------------------------------------*/
	if((mtx3354_status = mtx3354_horizontalSetup (instrSession, timebase, -250.0)) < 0)
		return mtx3354_status;

	/*- Setting trigger mode to AUTO-----------------------------------------*/
	/*- MTX3354 starts acquiring data immediately after it is triggered-------*/
	if((mtx3354_status = mtx3354_setGeneralTrigger (instrSession, channel, VI_ON, 0,VI_OFF, 1, 0.0, 40e-9)) < 0)
		return mtx3354_status;
        
	/*- Sending single-shot trigger in wait mode-----------------------------*/
	/*- This function will return as long as the acquisition is running------*/
	if((mtx3354_status = mtx3354_runAcquisition (instrSession, VI_ON)) < 0)
		return mtx3354_status;
        
	/*- Reads trace from the instrument--------------------------------------*/ 
	if((mtx3354_status = mtx3354_readChToArr (instrSession, channel, data, numberOfSamples, XStart, XIncrement)) < 0)
		return mtx3354_status; 
                   
	return mtx3354_status;
}
 
/*===========================================================================*/
/* Function: Configure MTX3354 Vertical Settings                             */
/* Purpose:  This function configures the vertical settings for a specified  */
/*           channel                                                         */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_verticalSetup (ViSession instrSession, ViInt16 input,
					ViBoolean state, ViInt16 coupling, ViInt16 probe, 
					ViInt16 sense, ViReal64 offset, ViReal64 band)
{
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(input, 1, 4))
		return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViBooleanRange(state))
		return VI_ERROR_PARAMETER3;
	if (mtx3354_invalidViInt16Range(coupling, 0, 2))
		return VI_ERROR_PARAMETER4;
	if (probe != 1 && probe != 10 && probe != 100 && probe != 1000)
		return VI_ERROR_PARAMETER5;
	if (mtx3354_invalidViInt16Range(sense, 0, 17))
		return VI_ERROR_PARAMETER6;
    
	switch (probe) 
		{
		/*--Probe x1---------------------------------------------------------*/
		case 1 :	if (mtx3354_invalidViInt16Range(sense,0,11))
						return VI_ERROR_MTX3354_INCONSISTENT_PARAMETER6;
                    break;
        /*--Probe x10--------------------------------------------------------*/
        case 10 :   if (mtx3354_invalidViInt16Range(sense,3,14))
                        return VI_ERROR_MTX3354_INCONSISTENT_PARAMETER6; 
					break;
		/*--Probe x100-------------------------------------------------------*/ 
		case 100 :	if (mtx3354_invalidViInt16Range(sense,6,17))
						return VI_ERROR_MTX3354_INCONSISTENT_PARAMETER6; 
					break;
		/*--Probe x1000------------------------------------------------------*/
		case 1000 :	if (mtx3354_invalidViInt16Range(sense,9,20))
						return VI_ERROR_MTX3354_INCONSISTENT_PARAMETER6; 
					break;  
		}      
	if (mtx3354_invalidViReal64Range(offset,-cmdOffsetMax[sense],cmdOffsetMax[sense]))
		return VI_ERROR_PARAMETER7;
    
	/*-Send vertical config command------------------------------------------*/
	Fmt (wrBuffer, "DISP:TRAC:Y:PDIV%i[b2] 1\r", input);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status; 
	Fmt (wrBuffer, "DISP:TRAC:STAT%i[b2] %i[b2]\r", input, state);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status; 
	Fmt (wrBuffer, "INP%i[b2]:COUP %s\r", input, cmdVertCoupling[coupling]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status; 
	Fmt (wrBuffer, "BAND%i[b2] %f[e3]\r", input, band);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status; 
	Fmt (wrBuffer, "VOLT%i[b2]:RANG:PTP %s\r", input, cmdVoltDiv[sense]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status; 
	Fmt (wrBuffer, "VOLT%i[b2]:RANG:OFFS %f\r", input, offset);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status; 
	Fmt (wrBuffer, "DISP:TRAC:Y:PDIV%i[b2] %i[b2]\r", input, probe);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status; 
	
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
        
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Configure MTX3354 Horizontal Settings                           */
/* Purpose:  This function configures the horizontal settings of the         */
/*           instrument                                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_horizontalSetup (ViSession instrSession,
					ViInt16 range, ViReal64 pretrg) 
                  
{
	ViStatus mtx3354_status = VI_SUCCESS;
	ViReal64 convBuf;
	ViByte pretrgBuf[BUFFER_SIZE];
	ViInt32 retCntptr;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(range, 0, 34))
		return VI_ERROR_PARAMETER2;
      
	/*----------------------- time bases ------------------------------------*/
	Fmt (pretrgBuf, "DISP:TRAC:X:PDIV %s\r", cmdTimeDiv[range]);
	if ((mtx3354_status = viWrite (instrSession, pretrgBuf, StringLength(pretrgBuf), &retCntptr)) < 0)
		return mtx3354_status; 
        
	/*-Check input parameter ranges and test pretrig-------------------------*/    
	Fmt(&convBuf, "%f<%s", cmdTimeDiv[range]);
	if (mtx3354_invalidViReal64Range(pretrg * cmdOffset[range],( -10 * convBuf), (25 * convBuf)))
		return VI_ERROR_PARAMETER3; 
  
	/*----------------  pretrig --------------------------------------------*/       
	Fmt (pretrgBuf, "SWE:OFFS:TIME %f\r", (pretrg * cmdOffset[range]));
	if ((mtx3354_status = viWrite (instrSession, pretrgBuf, StringLength(pretrgBuf), &retCntptr)) < 0)
		return mtx3354_status; 

	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;

	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Configure MTX3354 Horizontal Display                            */
/* Purpose:  This function configures the horizontal display                 */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_horizontalDisplay (ViSession instrSession, ViInt16 trace,
					ViInt16 x, ViInt16 y)
{
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(trace,0,1))
		return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViInt16Range(x,1,4))
		return VI_ERROR_PARAMETER3;
	if (mtx3354_invalidViInt16Range(y,1,4))
		return VI_ERROR_PARAMETER4;
   
	Fmt (wrBuffer,"DISP:TRAC:FORM %s;XY:XDEF INT%i[b2];YDEF INT%i[b2]\r",cmdDispTrace[trace],x,y);
	if ((mtx3354_status = viWrite (instrSession,wrBuffer,StringLength(wrBuffer),&retCnt)) < 0)
		return mtx3354_status;
        
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
     
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: FFT                                                             */
/* Purpose:  This function configures the FFT                                */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_FFT (ViSession instrSession, ViBoolean fftvalid, ViInt16 calculate, 
						ViBoolean scale)
{
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];     
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/

	if (mtx3354_invalidViBooleanRange(fftvalid))
		return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViInt16Range(calculate,0,5))
		return VI_ERROR_PARAMETER3;
	if (mtx3354_invalidViBooleanRange(scale))
		return VI_ERROR_PARAMETER4;
    
	/*-Set ON-OFF FFT, Calculate and scale parameters----------------*/ 
	Fmt (wrBuffer, "CALC:TRAN:FREQ %i[b2]\r", fftvalid);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
	Fmt (wrBuffer, "CALC:TRAN:FREQ:WIND %s\r", cmdcalcfft[calculate]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
	Fmt (wrBuffer, "DISP:TRAC:Y:SPAC %s\r", cmdScale[scale]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status; 
    
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;   
    
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: General Trigger                                                 */
/* Purpose:  This function sets general parameters for trigger               */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_setGeneralTrigger (ViSession instrSession, ViInt16 channel,
										ViBoolean slope, ViInt16 coupling, ViBoolean noiseReject,
										ViInt16 type, ViReal64 level, ViReal64 hold)
{
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(channel,0,4))
		return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViBooleanRange(slope))
		return VI_ERROR_PARAMETER3;
	if (mtx3354_invalidViInt16Range(coupling,0,5))
		return VI_ERROR_PARAMETER4;
	if (mtx3354_invalidViBooleanRange(noiseReject))
		return VI_ERROR_PARAMETER5;
	if (mtx3354_invalidViInt16Range(type,0,2))
		return VI_ERROR_PARAMETER6;
	if (channel < 4)
		{
		if (mtx3354_invalidViReal64Range(level,-800,800))
			return VI_ERROR_PARAMETER7;
		}
	else
		{ 
		if (mtx3354_invalidViReal64Range(level,-800E-3,8E-3))
			return VI_ERROR_PARAMETER7;
    	
		}
	if (mtx3354_invalidViReal64Range(hold,40E-9,10.5))
		return VI_ERROR_PARAMETER8;
    	
	/*-Set slope, coupling, level and filter parameters----------------*/
	Fmt (wrBuffer, "TRIG:SEQ1:SLOP %s\r", cmdSlope[slope]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
	if (channel < 4)
		{
		Fmt (wrBuffer, "TRIG:LEV %f\r", level);
		if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3354_status;
		Fmt (wrBuffer,"TRIG:HOLD %f\r",hold);
		if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3354_status; 
		Fmt (wrBuffer, "TRIG:HYST %i\r", noiseReject);
		if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3354_status;
		switch (coupling)
			{
			case 0 :
					Fmt (wrBuffer, "TRIG:COUP %s\r", cmdTrigCoupling[coupling]);
					if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
						return mtx3354_status;
					break;
			case 1 :
					Fmt (wrBuffer, "TRIG:COUP %s\r", cmdTrigCoupling[coupling]);
					if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
						return mtx3354_status;
					break;	
			case 2 :
					Fmt (wrBuffer, "TRIG:FILT:HPAS 1\r");
					if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
						return mtx3354_status;
					break;	
			case 3 :
					Fmt (wrBuffer, "TRIG:FILT:LPAS 1\r");
					if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
						return mtx3354_status;
					break;	
			}
		}

	/*-Set trigger type and mode parameters----------------*/
	TrigMode = type;
	if (type == 2)
		{
		if (channel == 4)
			Fmt (wrBuffer, ":INIT:NAME LINE\r");
		else
			Fmt (wrBuffer, ":INIT:NAME EDGE\rTRIG:SOUR %s\r", cmdSource[channel]);
		
		if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3354_status; 
		}
	else
		{
		if (channel == 4)
			Fmt (wrBuffer, ":INIT:CONT:NAME LINE,1\r");
		else
			Fmt (wrBuffer, ":INIT:CONT:NAME EDGE,1\rTRIG:SOUR %s\r", cmdSource[channel]);
		
		if (type == 1)
			Fmt (wrBuffer, "%sTRIG:SEQ1:ATRIG 1\r", wrBuffer);
			
		if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3354_status; 
		}
        
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;   
    
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: PULSE Trigger                                                   */
/* Purpose:  This function sets Pulsate trigger type                         */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_setPulsTrigger (ViSession instrSession, ViInt16 pulsType,
                    		ViReal64 pulse)
{                                
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViBoolean runStat;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(pulsType,0,2))
		return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViReal64Range(pulse,10E-9,10.5))
		return VI_ERROR_PARAMETER3;

   	if(TrigMode == 0)
		{
		Fmt (rdBuffer ,"TRIG:RUN:STAT?\r");
		if ((mtx3354_status = viWrite (instrSession,rdBuffer,StringLength(rdBuffer), &retCnt)) < 0)
			return mtx3354_status;
		if ((mtx3354_status = viScanf (instrSession,"%hd", &runStat)) < 0)
			return mtx3354_status;
		Fmt (rdBuffer, "INIT:CONT:NAME PUL,%i[b1]\r", runStat);
		if ((mtx3354_status = viWrite (instrSession, rdBuffer, StringLength(rdBuffer), &retCnt)) < 0)
			return mtx3354_status;
		}
	 else if(TrigMode == 2)
	 	{
		Fmt (rdBuffer, "INIT:NAME PUL\r");
		if ((mtx3354_status = viWrite (instrSession, rdBuffer, StringLength(rdBuffer), &retCnt)) < 0)
			return mtx3354_status; 
	 	}
    
	/*-Set source and levels for edge trigger----------------------------*/
	if ((mtx3354_status = viPrintf (instrSession, "TRIG:TYP %s\r", CmdPulsTyp[pulsType])) < 0)
		return mtx3354_status;
	Fmt (rdBuffer,"TRIG:SEQ2:DEL %f\r",pulse);
	if ((mtx3354_status = viWrite (instrSession, rdBuffer, StringLength(rdBuffer), &retCnt)) < 0)
		return mtx3354_status; 

	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;

	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Delayed Trigger                                                 */
/* Purpose:  This function sets delayed trigger type                         */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_setDelayedTrigger (ViSession instrSession, ViInt16 auxi_source, ViBoolean edge,
							ViInt16 coupling, ViInt16 noiseReject, ViReal64 level, ViReal64 delay, ViReal64 hold)
{                                
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViByte runStat;

    
    /*-Check input parameter ranges------------------------------------------*/
    if (mtx3354_invalidViInt16Range(auxi_source,1,4))
        return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViBooleanRange(edge))
		return VI_ERROR_PARAMETER3;
	if (mtx3354_invalidViInt16Range(coupling,0,3))
		return VI_ERROR_PARAMETER4;
	if (mtx3354_invalidViInt16Range(noiseReject,0,3))
		return VI_ERROR_PARAMETER5;
    if (mtx3354_invalidViReal64Range(level,-800,800))
        return VI_ERROR_PARAMETER5;
    if (mtx3354_invalidViReal64Range(delay,20.0E-9,10.5))
        return VI_ERROR_PARAMETER6;
	if (mtx3354_invalidViReal64Range(hold,40E-9,10.5))
		return VI_ERROR_PARAMETER7;
        
    /*-Set delay parameters--------------------------------------------------*/
	if ((mtx3354_status = viPrintf (instrSession,"ARM:SOUR INT%hd\r", auxi_source)) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viPrintf (instrSession,"ARM:SLOP %s\r", cmdSlope[edge])) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viPrintf (instrSession,"ARM:LEV %lf\r", level)) < 0)
		return mtx3354_status;
	Fmt (wrBuffer,"TRIG:SEQ3:DEL %f\r",delay);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status; 
	switch (coupling)
		{
		case 0 :
				Fmt (wrBuffer, "ARM:COUP %s\r", cmdTrigCoupling[coupling]);
				if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
					return mtx3354_status;
				break;
		case 1 :
				Fmt (wrBuffer, "ARM:COUP %s\r", cmdTrigCoupling[coupling]);
				if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
					return mtx3354_status;
				break;	
		case 2 :
				Fmt (wrBuffer, "ARM:FILT:HPAS 1\r");
				if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
					return mtx3354_status;
				break;	
		case 3 :
				Fmt (wrBuffer, "ARM:FILT:LPAS 1\r");
				if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
					return mtx3354_status;
				break;	
		}
	Fmt (wrBuffer,"TRIG:HOLD %f\r",hold);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status; 
	Fmt (wrBuffer, "TRIG:HYST %i\r", noiseReject);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
		
	if(TrigMode == 0)
		{
		Fmt (wrBuffer ,"TRIG:RUN:STAT?\r");
		if ((mtx3354_status = viPrintf (instrSession,wrBuffer,StringLength(wrBuffer))) < 0)
			return mtx3354_status;
		if ((mtx3354_status = viScanf (instrSession,"%hd", &runStat)) < 0)
			return mtx3354_status;
		Fmt (wrBuffer, "INIT:CONT:NAME DEL,%i[b1]\r", runStat);
		if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3354_status;
		}
	 else if(TrigMode == 2)
	 	{
		Fmt (wrBuffer, "INIT:NAME DEL\r");
		if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3354_status; 
	 	}
	
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;

	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Count Trigger                                                   */
/* Purpose:  This function sets event-count trigger type                     */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_setCountTrigger (ViSession instrSession, ViInt16 auxi_source,
					ViBoolean edge,ViInt16 coupling, ViInt16 noiseReject, ViReal64 level, ViInt32 events, ViReal64 hold)
{                                
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViByte runStat = 1;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(auxi_source,1,4))
		return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViBooleanRange(edge))
		return VI_ERROR_PARAMETER3;
	if (mtx3354_invalidViInt16Range(coupling,0,3))
		return VI_ERROR_PARAMETER4;
	if (mtx3354_invalidViInt16Range(noiseReject,0,3))
		return VI_ERROR_PARAMETER5;
	if (mtx3354_invalidViReal64Range(level,-800,800))
		return VI_ERROR_PARAMETER5;
	if (mtx3354_invalidViInt32Range(events,3,16384))
		return VI_ERROR_PARAMETER6;
	if (mtx3354_invalidViReal64Range(hold,40E-9,10.5))
		return VI_ERROR_PARAMETER7;

	/*-Set count trigger parameters------------------------------------------*/
	if ((mtx3354_status = viPrintf (instrSession,"ARM:SOUR INT%hd\r", auxi_source)) < 0)
		return mtx3354_status;

	if ((mtx3354_status = viPrintf (instrSession,"ARM:SLOP %s\r", cmdSlope[edge])) < 0)
		return mtx3354_status;

	if ((mtx3354_status = viPrintf (instrSession,"ARM:LEV %lf\r", level)) < 0)
		return mtx3354_status;

	if ((mtx3354_status = viPrintf (instrSession,"TRIG:ECO %ld\r",events)) < 0)
		return mtx3354_status;
	switch (coupling)
		{
		case 0 :
				Fmt (wrBuffer, "ARM:COUP %s\r", cmdTrigCoupling[coupling]);
				if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
					return mtx3354_status;
				break;
		case 1 :
				Fmt (wrBuffer, "ARM:COUP %s\r", cmdTrigCoupling[coupling]);
				if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
					return mtx3354_status;
				break;	
		case 2 :
				Fmt (wrBuffer, "ARM:FILT:HPAS 1\r");
				if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
					return mtx3354_status;
				break;	
		case 3 :
				Fmt (wrBuffer, "ARM:FILT:LPAS 1\r");
				if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
					return mtx3354_status;
				break;	
		}
	Fmt (wrBuffer,"TRIG:HOLD %f\r",hold);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status; 
	Fmt (wrBuffer, "TRIG:HYST %i\r", noiseReject);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
		
	if(TrigMode == 0)
		{
		Fmt (wrBuffer ,"TRIG:RUN:STAT?\r");
		if ((mtx3354_status = viPrintf (instrSession,wrBuffer,StringLength(wrBuffer))) < 0)
			return mtx3354_status;
		if ((mtx3354_status = viScanf (instrSession,"%hd", &runStat)) < 0)
			return mtx3354_status;
		Fmt (wrBuffer, "INIT:CONT:NAME EVEN,%i[b1]\r", runStat);
		if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3354_status;
		}
	else if(TrigMode == 2)
	 	{
		Fmt (wrBuffer, "INIT:NAME EVEN\r");
		if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3354_status; 
	 	}
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;

	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Video Trigger                                                   */
/* Purpose:  This function sets video trigger type                           */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_setVideoTrigger (ViSession instrSession, ViBoolean standard,
								ViBoolean polarity, ViInt16 line, ViReal64 hold)
{                                
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViByte runStat = 1;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViBooleanRange(standard))
		return VI_ERROR_PARAMETER3;
	if (mtx3354_invalidViBooleanRange(polarity))
		return VI_ERROR_PARAMETER4;
	if (mtx3354_invalidViInt16Range(line,1,625))
		return VI_ERROR_PARAMETER5;
	if (mtx3354_invalidViReal64Range(hold,40E-9,10.5))
		return VI_ERROR_PARAMETER6;

	/*-Set Video trigger parameters------------------------------------------*/
	if ((mtx3354_status = viPrintf (instrSession,"TRIG:VID:LINE:SEL %hd\r",line)) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viPrintf (instrSession,"TRIG:VID:SSIG %s\r",cmdSlopeVid[polarity])) < 0)
		return mtx3354_status;
	Fmt (wrBuffer,"TRIG:HOLD %f\r",hold);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status; 
	Fmt (wrBuffer,"TRIG:VID:FIEL:FORM:LPFR %s\r",cmdStandVid[standard]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status; 

	if(TrigMode == 0)
		{
		Fmt (wrBuffer ,"TRIG:RUN:STAT?\r");
		if ((mtx3354_status = viPrintf (instrSession,wrBuffer,StringLength(wrBuffer))) < 0)
			return mtx3354_status;
		if ((mtx3354_status = viScanf (instrSession,"%hd", &runStat)) < 0)
			return mtx3354_status;
		Fmt (wrBuffer, "INIT:CONT:NAME TV,%i[b1]\r", runStat);
		if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3354_status;
		}
	else if(TrigMode == 2)
	 	{
		Fmt (wrBuffer, "INIT:NAME TV\r");
		if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3354_status; 
	 	}
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
        
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Display Options                                                 */
/* Purpose:  This function sets display options.                             */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_setOptionDisplay (ViSession instrSession, ViBoolean grid,
					ViInt16 device_mode, ViBoolean avanced_mode, ViInt16 language, 
					ViInt16 standby, ViInt16 screen_saver, ViInt16 brightness, ViInt16 contrast)
{
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViBooleanRange(grid))
		return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViInt16Range(device_mode,0,3))
		return VI_ERROR_PARAMETER3;
	if (mtx3354_invalidViBooleanRange(avanced_mode))
		return VI_ERROR_PARAMETER4;
	if (mtx3354_invalidViInt16Range(language,0,4))
		return VI_ERROR_PARAMETER5;
	if (mtx3354_invalidViInt16Range(standby,0,3))
		return VI_ERROR_PARAMETER6;
	if (mtx3354_invalidViInt16Range(screen_saver,0,3))
		return VI_ERROR_PARAMETER7;
	if (mtx3354_invalidViInt16Range(brightness,0,100))
		return VI_ERROR_PARAMETER8;
	if (mtx3354_invalidViInt16Range(contrast,0,100))
		return VI_ERROR_PARAMETER9;

	/*-Set option display parameters-----------------------------------------*/
	Fmt (wrBuffer, "DISP:TRAC:GRAT:GRID %i[b2]\r", grid);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
	Fmt (wrBuffer, "DEV:MOD %s\r", cmddevice[device_mode]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
	if (device_mode == 3)
		{
		Fmt (wrBuffer, "DISP:PERS 1\r");
		if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3354_status;
		}
	Delay (2.0);
	 
	Fmt (wrBuffer, "DEV:SCOP:EXP %i[b2]\r", avanced_mode);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
	Delay (2.0);
	        
	Fmt (wrBuffer, "SYST:LANG %s\r", cmdDisplayLang[language]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
	   return mtx3354_status;      
	Delay (2.0);
	  
	Fmt (wrBuffer, "DISP:SCRE:SAVER %s\r", cmdDisplayScreen[screen_saver]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;   
	Fmt (wrBuffer, "SYST:POW:OFF %s\r", cmdSystemStandby[standby]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;   		
	Fmt (wrBuffer, "DISP:BRIG %f[p2]\r", brightness / 100.0);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
	Fmt (wrBuffer, "DISP:CONT %f[p2]\r", contrast / 100.0);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status; 		

	/*-Check instrument status & Return from function------------------------*/				   
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
      
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Display Waveform                                                */
/* Purpose:  This function sets waveform display.                            */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_setWaveformDisplay (ViSession instrSession,ViInt16 acquisition,
					ViInt16 average_rate, ViInt16 mode, ViBoolean repet)
{
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(acquisition, 0,1))
		return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViInt16Range(average_rate, 0, 4))
		return VI_ERROR_PARAMETER3;
	if (mtx3354_invalidViInt16Range(mode, 0, 1))
		return VI_ERROR_PARAMETER4;
	if (mtx3354_invalidViBooleanRange(repet))
		return VI_ERROR_PARAMETER5;

	/*-Set waveform display parameters---------------------------------------*/
	Fmt (wrBuffer, "AVER:TYPE %s\r", cmdMaxminAcq[acquisition]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
	Fmt (wrBuffer, "AVER %i[b2]\r", repet);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
	Fmt (wrBuffer, "DISP:TRACE:MODE %s\r", cmdMode[mode]);
		if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3354_status;
	if(repet==1)
		{
		Fmt (wrBuffer, "AVER:COUN %s\r", cmdAverage[average_rate]);
		if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3354_status;
    	}
	
    /*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
        
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Set SPO options                                                 */
/* Purpose:  This function sets the SPO options.                             */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_setSPOOptions (ViSession instrSession, ViInt16 palette, ViInt16 timing)
{
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(palette,0,1))
		return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViInt16Range(timing,0,7))
		return VI_ERROR_PARAMETER3;

	/*-Set palette-----------------------------------------------------------*/    
	Fmt (wrBuffer, "DISP:TRAC:PERS:PALE %s\r", cmdPalette[palette]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
    
	/*-Set timing------------------------------------------------------------*/    
	Fmt (wrBuffer, "DISP:TRAC:PERS:TIME %s\r", cmdTiming[timing]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
    
	return mtx3354_status;

}

/*===========================================================================*/
/* Function: Set Cursors                                                     */
/* Purpose:  This function sets MTX3354 cursors.                              */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_setCursors (ViSession instrSession, ViInt16 cursor,
					ViInt16 reference, ViBoolean unattcur, ViReal64 t1, ViReal64 t2, ViReal64 tphase)
{                              
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(cursor,0,2))
		return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViInt16Range(reference,0,1))
		return VI_ERROR_PARAMETER3;
	if ((mtx3354_invalidViBooleanRange(unattcur))) 
		return VI_ERROR_PARAMETER4;
	if (mtx3354_invalidViReal64Range(t1,0,2000.0))
		return VI_ERROR_PARAMETER5;
	if (mtx3354_invalidViReal64Range(t2,0,2000.0))
		return VI_ERROR_PARAMETER6;
	if (mtx3354_invalidViReal64Range(tphase,0,2000))
		return VI_ERROR_PARAMETER7;    
	 
	/*-Set cursors-----------------------------------------------------------*/    
	Fmt (wrBuffer, "DISP:CURS:REF %s\r", cmdRef[reference]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
    
	switch(cursor)
		{
		case 0 :
			Fmt (wrBuffer, "DISP:CURS:STAT OFF;PHASE:STAT OFF\r");
			if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
				return mtx3354_status;
			break;
			
		case 1 :
			Fmt (wrBuffer, "DISP:CURS:STAT ON;PHASE:STAT OFF\r");
			if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
				return mtx3354_status;
			if ((mtx3354_status = viPrintf (instrSession,"DISP:CURS:TIME1:POS %lf\r",t1)) < 0)
				return mtx3354_status;
			Delay (1.7);
            	
			if ((mtx3354_status = viPrintf (instrSession,"DISP:CURS:TIME2:POS %lf\r",t2)) < 0)
				return mtx3354_status;
			break;
		
		case 2 :
			Fmt (wrBuffer, "DISP:CURS:STAT OFF;PHASE:STAT ON\r");
			if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
				return mtx3354_status;
			if ((mtx3354_status = viPrintf (instrSession,"DISP:CURS:TIME1:POS %lf\r",t1)) < 0)
				return mtx3354_status;
			Delay (1.7);	
            	
			if ((mtx3354_status = viPrintf (instrSession,"DISP:CURS:TIME2:POS %lf\r",t2)) < 0)
				return mtx3354_status;
			if ((mtx3354_status = viPrintf (instrSession,"DISP:CURS:TIME3:POS %lf\r",tphase)) < 0)
				return mtx3354_status;	
			break;	
    	
		}
	Fmt (wrBuffer, "DISP:CURS:AUTO:STAT %i[b2]\r", unattcur);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;

	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
        
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Hardcopy                                                        */
/* Purpose:  This function configures MTX3354 hardcopy facility.              */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_setHardcopy (ViSession instrSession, ViInt16 device,
					ViInt16 port, ViString filename, ViBoolean type)
{                              
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
            
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(device,0,7))
		return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViInt16Range(port,0,3))
		return VI_ERROR_PARAMETER3;
	if (mtx3354_invalidViBooleanRange(type))
		return VI_ERROR_PARAMETER5;
	    
	/*-Set hardcopy parameters-----------------------------------------------*/
	if (port == 3)
		{
		if (StringLength(filename) != 8) 
			return VI_ERROR_PARAMETER4;
		if ((mtx3354_status = viPrintf (instrSession,"HCOP:DEST \"%s\"\r",filename)) < 0)
			return mtx3354_status;
		if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
		}	
	Fmt (wrBuffer, "HCOP:DEST \"%s\"\r",cmdDestination[port]);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
	Fmt (wrBuffer, "HCOP:DEV:LANG %s; COL %i[b2]\r", cmdHardcopyLang[device], type);
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;
	
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
        
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Run Autoset                                                     */
/* Purpose:  This function performs an automatic setup of the MTX3354.       */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_runAutoset (ViSession instrSession)
{                              
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
    
	Fmt (wrBuffer, "AUTOS:EXE\r");
	if ((mtx3354_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3354_status;

	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
        
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Run Acquisition                                                 */
/* Purpose:  This function performs an acquisition session.                  */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_runAcquisition (ViSession instrSession, ViBoolean run)
{                              
	ViStatus mtx3354_status = VI_SUCCESS;
	ViInt32 retCnt;
	ViByte rdBuffer[BUFFER_SIZE];
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViBooleanRange(run))
		return VI_ERROR_PARAMETER3;
    
	/*-Get current trigger mode----------------------------------------------*/
	Fmt (rdBuffer, "TRIG:RUN:STAT %i[b2]\r", run);
	if ((mtx3354_status = viWrite (instrSession, rdBuffer, StringLength(rdBuffer), &retCnt)) < 0)
		return mtx3354_status;
		
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
        
    return mtx3354_status;
}

/*===========================================================================*/
/* Function: Run Hardcopy                                                    */
/* Purpose:  This function performs a hardcopy.                              */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_runHardcopy (ViSession instrSession, ViBoolean action)
{                              
	ViStatus mtx3354_status = VI_SUCCESS;
    
	if (action) 
		{               
		if ((mtx3354_status = viPrintf (instrSession,"HCOP:SDUM\r")) < 0)
			return mtx3354_status;
		}
	else {
		if ((mtx3354_status = viPrintf (instrSession,"HCOP:SDUM:UNDO\r")) < 0)
			return mtx3354_status;
	}   
        
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Read Cursors                                                    */
/* Purpose:  This function reads MTX3354 cursors.                             */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_readCursors (ViSession instrSession, ViInt16 reference,
					ViPReal64 t1, ViPReal64 t2, ViPReal64 dt, ViPReal64 tphase, ViPReal64 v1, 
					ViPReal64 v2, ViPReal64 dv, ViPReal64 vphase)
{                              
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViInt16 valtphase;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(reference,0,1))
		return VI_ERROR_PARAMETER2;

	/*-Display cursor menu---------------------------------------------------*/
	Fmt (rdBuffer, "DISP:CURS:REF %s\r", cmdRef[reference]);
	if ((mtx3354_status = viWrite (instrSession, rdBuffer, StringLength(rdBuffer), &retCnt)) < 0)
		return mtx3354_status;
        
	/*-Set cursor positions--------------------------------------------------*/
	if ((mtx3354_status = viPrintf (instrSession,"DISP:CURS:TIME1:POS?\r")) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viScanf (instrSession,"%lf",t1)) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viPrintf (instrSession,"DISP:CURS:TIME2:POS?\r")) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viScanf (instrSession,"%lf",t2)) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viPrintf (instrSession,"DISP:CURS:PHAS:STAT?\r")) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viScanf (instrSession, "%i", &valtphase)) < 0)
		return mtx3354_status;
	if(valtphase == 1)	
		{
		if ((mtx3354_status = viPrintf (instrSession,"DISP:CURS:TIME3:POS?\r")) < 0)
			return mtx3354_status;
		if ((mtx3354_status = viScanf (instrSession,"%lf",tphase)) < 0)
			return mtx3354_status;
		if ((mtx3354_status = viPrintf (instrSession,"DISP:CURS:VOLT3:POS?\r")) < 0)
			return mtx3354_status;
		if ((mtx3354_status = viScanf (instrSession,"%lf",vphase)) < 0)
			return mtx3354_status;
		}
	if ((mtx3354_status = viPrintf (instrSession,"DISP:CURS1:VOLT:POS?\r")) < 0)
		return mtx3354_status;

	if ((mtx3354_status = viScanf (instrSession,"%lf",v1)) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viPrintf (instrSession,"DISP:CURS:VOLT2:POS?\r")) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viScanf (instrSession,"%lf",v2)) < 0)
		return mtx3354_status;
		
	/*-Calculate 'delta t' & 'delta v'---------------------------------------*/
	*dt = *t2 - *t1;
	*dv = *v2 - *v1;
        
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
        
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Read Measurement                                                */
/* Purpose:  This function reads a specified measurement.                    */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_readMeasurement (ViSession instrSession, ViInt16 measure,
					ViInt16 reference, ViPReal64 reading)
{                              
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(measure,0,18))
		return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViInt16Range(reference,0,1))
		return VI_ERROR_PARAMETER3;
	Fmt (rdBuffer, "DISP:CURS:REF %s\r", cmdRef[reference]);
	if ((mtx3354_status = viWrite (instrSession, rdBuffer, StringLength(rdBuffer), &retCnt)) < 0)
		return mtx3354_status;
	if(measure<18) 
		{
		if ((mtx3354_status = viPrintf (instrSession,"MEAS:%s? %s\r",cmdReadMeasure[measure],cmdRef[reference])) < 0)
			return mtx3354_status;
		}
	else
		{
		if ((mtx3354_status = viPrintf (instrSession,"MEAS:PHAS? %s,%s\r", cmdRef[reference] ,cmdRef[!reference])) < 0)
			return mtx3354_status;
		}	
	if ((mtx3354_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return mtx3354_status;
	if ((mtx3354_status = Scan (rdBuffer,"%s>%f",reading))!=1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
     
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
        
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Read Channel To Array                                           */
/* Purpose:  This function reads a trace from the instrument.                */
/*           Data is stored in an array                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_readChToArr (ViSession instrSession, ViInt16 source,			   
					ViReal64 _VI_FAR waveformArray[], ViPInt32 numberOfSamples, 
					ViPReal64 XStart, ViPReal64 XIncrement)
{   
	ViStatus mtx3354_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViInt32 retCnt,size, index;
	ViInt16 tstRes;
	ViUInt16 interface;
	static ViByte tabByte[MAX_MEM_SIZE];
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(source,0,1))
		return VI_ERROR_PARAMETER2;
    
	/*-Check if source is active---------------------------------------------*/
	if ((mtx3354_status = viPrintf (instrSession,":DISP:TRAC:STAT%hd?\r", source + 1)) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viRead (instrSession, tabByte, 5, &retCnt)) < 0)
		return mtx3354_status;
	Scan (tabByte,"%i[b2]", &tstRes);
	if (tstRes != 1) 
		return VI_ERROR_MTX3354_CHAN_NOT_ACTIVE;
    
	/*-Modify Time Out for big waveform transfers------------------------*/
	if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_TRANSFER)) < 0)
		return mtx3354_status;

	/*-Modify Protocol for binary file transfers-------------------------*/
	viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
	if (interface == VI_INTF_ASRL)
	{
		if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
			return mtx3354_status;
	}

	/*-Set data format and start transfert-----------------------------------*/    
	if ((mtx3354_status = viPrintf (instrSession,"TRAC? INT%hd\r", source + 1)) < 0)
		return mtx3354_status;
	if (interface==VI_INTF_ASRL) 
	{
		/* Read 11 characters (#an header) to define the length of the binary block following */
		if ((mtx3354_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
			return mtx3354_status;
		index = FindPattern (tabByte, 0, HEADER_SIZE, "#", 0, 0);
		headerSize = tabByte[index + 1] - 0x30;
		Scan(tabByte + index + 2, "%i[b4]", &size);
		if ((mtx3354_status = viRead (instrSession, tabByte + HEADER_SIZE, index + size + 7 + headerSize - HEADER_SIZE, &retCnt)) < 0)
			return mtx3354_status;
		/*-Reset Protocol for binary file transfers----------------------------*/
		if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
			return mtx3354_status;
	}
	else if ((mtx3354_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
		return mtx3354_status;

	/*-Reset Time Out ---------------------------------------------------*/
	if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
		return mtx3354_status;
		
	/*-Make data readable----------------------------------------------------*/
	if ((mtx3354_status = mtx3354_makeDataReadable(tabByte,waveformArray,numberOfSamples,XStart,XIncrement)) < 0)
		return mtx3354_status;
                            
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
        
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Read Channel To File                                            */
/* Purpose:  This function reads a trace from the instrument.                */
/*           Data is stored in a file                                        */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_readChToFile ( ViSession instrSession, ViInt16 channel,
                    ViInt16 drive, ViString filename, ViBoolean overwrite)
{                        
	ViStatus mtx3354_status = VI_SUCCESS;
	ViInt32 fileSize, retCnt, RunState;													
	static ViByte tabByte[MAX_MEM_SIZE];
    ViInt16 tstRes;
    ViString filenameTemp = "___tmp__.trc";

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(channel,0,1))
		return VI_ERROR_PARAMETER2;								 
	if (mtx3354_invalidViInt16Range(drive,0,1))
		return VI_ERROR_PARAMETER3;
	if (mtx3354_invalidViBooleanRange(overwrite))
		return VI_ERROR_PARAMETER5;
        
	/*-Check if source is active---------------------------------------------*/
	if ((mtx3354_status = viPrintf (instrSession,":DISP:TRAC:STAT%hd?\r", channel + 1)) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viRead (instrSession, tabByte, 5, &retCnt)) < 0)
		return mtx3354_status;
	Scan (tabByte,"%i[b2]", &tstRes);
	if (tstRes != 1) 
		return VI_ERROR_MTX3354_CHAN_NOT_ACTIVE;
    
	/*-Check filename--------------------------------------------------------*/
	if (mtx3354_invalidFilename (filename,drive,".TRC"))
		return VI_ERROR_PARAMETER4;

	/* if oscilloscope is in Run mode ?*/
	viWrite (instrSession, "TRIG:RUN:STAT?\r", 15, &RunState);
	viScanf (instrSession, "%hd", &RunState);
	if (RunState)
		viWrite (instrSession, "TRIG:RUN:STAT 0\r", 16, &RunState);
	
	/*-Store current trace into MTX3354---------------------------------------*/
	if (drive) 
		{
		/*-Check overwrite---------------------------------------------------*/
		if ((mtx3354_status = viPrintf (instrSession,"MMEM:CAT?\r")) < 0)
			return mtx3354_status;
		if ((mtx3354_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mtx3354_status;
		if (FindPattern (tabByte, 0, -1, filename, 0, 0) != -1) 
			{
			if (overwrite) 
				{
				if ((mtx3354_status = viPrintf (instrSession,"MMEM:DEL \"%s\"\r", filename)) < 0)
					return mtx3354_status;
				}
			else 
				return VI_ERROR_MTX3354_EXISTING_FILE;
			}
        
		/*-Perform transfer--------------------------------------------------*/
		if ((mtx3354_status = viPrintf (instrSession,"MMEM:STOR:TRAC INT%hd,\"%s\"\r",
					channel+1,filename )) < 0)
			return mtx3354_status;
		}
    
	/*-Store current trace into a PC file------------------------------------*/
	else 
		{
		/*-Check overwrite---------------------------------------------------*/
		if (GetFileInfo(filename,&fileSize)  && !overwrite) 
			return VI_ERROR_MTX3354_EXISTING_FILE;

		if (GetFileInfo (filename,&fileSize)  && overwrite) 
			DeleteFile (filename);

		/*-Perform transfer--------------------------------------------------*/
		if ((mtx3354_status = viPrintf (instrSession,"MMEM:STOR:TRAC INT%hd,\"%s\"\r",
					channel+1, filenameTemp )) < 0)
			return mtx3354_status;

		/*-Temp file transfer------------------------------------------------*/
		if ((mtx3354_status = mtx3354_fileTransfer (instrSession, filenameTemp, 1, VI_ON)) < 0)
			return mtx3354_status;

		/*-Rename Temp file--------------------------------------------------*/
		RenameFile (filenameTemp, filename);
			
		/*-Delete Temp file--------------------------------------------------*/
		if ((mtx3354_status = viPrintf (instrSession,"MMEM:DEL \"%s\"\r", filenameTemp)) < 0)
			return mtx3354_status;
		}

	/*Restart the oscilloscope in Run mode*/
	if (RunState)
		viWrite (instrSession, "TRIG:RUN:STAT 1\r", 16, &RunState);
  
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
        
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Read File To Array                                              */
/* Purpose:  This function reads a file from the PC or from the instrument   */
/*           Data is stored in an array                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_readFileToArr (ViSession instrSession, ViInt16 drive,
					ViString filename, ViReal64 _VI_FAR waveformArray[], ViPInt32 numberOfSamples, 
					ViPReal64 XStart, ViPReal64 XIncrement)
{   
	ViStatus mtx3354_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViInt32 retCnt,fileSize, size, index;
	static ViByte tabByte[MAX_MEM_SIZE];
	ViUInt16 interface;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(drive,0,1))
		return VI_ERROR_PARAMETER2;
	        
	/*-Check filename--------------------------------------------------------*/
	if (mtx3354_invalidFilename (filename,drive,".TRC"))
		return VI_ERROR_PARAMETER3;
        
	/*-Read file from MTX3354-------------------------------------------------*/
	if (drive) 
		{
		/*-Check filename----------------------------------------------------*/
		if ((mtx3354_status = viPrintf (instrSession,"MMEM:CAT?\r")) < 0)
			return mtx3354_status;
		if ((mtx3354_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mtx3354_status;
		if (FindPattern (tabByte, 0, -1, filename, 0, 0) < 0)
			return VI_ERROR_MTX3354_FILE_NOT_FOUND;
        
		/*-Modify Time Out for big waveform transfers------------------------*/
		if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_TRANSFER)) < 0)
			return mtx3354_status;
	
		/*-Modify Protocol for binary file transfers-------------------------*/
		viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
		if(interface==VI_INTF_ASRL)
			{
			if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
				return mtx3354_status;
			}

		/*-Perform transfer--------------------------------------------------*/
		if ((mtx3354_status = viPrintf (instrSession,"MMEM:DATA? \"%s\"\r",filename)) < 0)
			return mtx3354_status;
		if (interface==VI_INTF_ASRL) 
			{
			/* Read 11 characters (#an header) to define the length of the binary block following */
			if ((mtx3354_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
				return mtx3354_status;
			index = FindPattern (tabByte, 0, -1, "#", 0, 0);
			headerSize = tabByte[index + 1] - 0x30;
			Scan(tabByte + index + 2, "%i[b4]", &size);
			if ((mtx3354_status = viRead (instrSession, tabByte + HEADER_SIZE, size + 4 + headerSize - HEADER_SIZE, &retCnt)) < 0)
				return mtx3354_status;
				
			retCnt += HEADER_SIZE;

			/*-Reset Protocol for binary file transfers----------------------------*/
			if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
				return mtx3354_status;
			}
				
		/*-Reset Time Out ---------------------------------------------------*/
		if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE,TMO_VALUE)) < 0)
			return mtx3354_status;
    	}
	/*-Read file from PC-----------------------------------------------------*/
	else 
		{
		/*-Check filename----------------------------------------------------*/
		if (!GetFileInfo(filename,&fileSize)) 
			return VI_ERROR_MTX3354_FILE_NOT_FOUND;
        if ((FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, MAX_MEM_SIZE,MAX_MEM_SIZE,
			VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_BINARY)) < 0)
		return VI_ERROR_INSTR_FILE_OPEN;
		}
		
	/*-Make data readable----------------------------------------------------*/
	if ((mtx3354_status = mtx3354_makeDataReadable(tabByte,waveformArray,numberOfSamples, XStart, XIncrement)) < 0)
		return mtx3354_status;
    
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
        
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Write File To Instrument Display Memory                         */
/* Purpose:  This function fills the instrument display memory.              */
/*           Data are in the specified file.                                 */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_writeFileToMem(ViSession instrSession, ViString filename, ViInt16 memory)
{   
	ViStatus mtx3354_status = VI_SUCCESS;
	ViUInt32 retCnt;
	static ViByte tabByte[MAX_MEM_SIZE];

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(memory,1,4))
		return VI_ERROR_PARAMETER4;

	/*-Check filename--------------------------------------------------------*/
	if (mtx3354_invalidFilename (filename,1,".TRC"))
		return VI_ERROR_PARAMETER3;
    
	/*-Check filename----------------------------------------------------*/
	if ((mtx3354_status = viPrintf (instrSession,"MMEM:CAT?\r")) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
		return mtx3354_status;
	if (FindPattern (tabByte, 0, -1, filename , 0, 0) < 0)
		return VI_ERROR_MTX3354_FILE_NOT_FOUND;

	/*-Perform transfer--------------------------------------------------*/
	if ((mtx3354_status = viPrintf (instrSession,"MMEM:LOAD:TRAC TRACE%hd,\"%s\"\r",
	            memory, filename)) < 0)
		return mtx3354_status;
		
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
    
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Save/Recall Setup From/To Instrument                            */
/* Purpose:  This function allows you to save the current setup of the       */
/*           instrument or to recall a setup stored in a file.               */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_saveSetup (ViSession instrSession, ViBoolean mode, ViInt16 drive,
				ViString filename, ViBoolean overwrite)
{
	ViStatus mtx3354_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViUInt32 retCnt, fileSize;
	static ViByte string[2000],tabByte[MAX_MEM_SIZE];
	ViUInt16 interface;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViBooleanRange(mode))
		return VI_ERROR_PARAMETER2;
	if (mtx3354_invalidViInt16Range(drive,0,1))
		return VI_ERROR_PARAMETER3;
	if (mtx3354_invalidViBooleanRange(overwrite))
		return VI_ERROR_PARAMETER5;

	/*-Check filename--------------------------------------------------------*/
	if (mtx3354_invalidFilename (filename,drive,".CFG"))
		return VI_ERROR_PARAMETER4;

	/*-Save setup from MTX3354------------------------------------------------*/
	if (mode) 	
		{
		/*-Target drive is MTX3354--------------------------------------------*/
		if (drive) 
			{
			/*-Check overwrite-----------------------------------------------*/
			if ((mtx3354_status = viPrintf (instrSession,"MMEM:CAT?\r")) < 0)
				return mtx3354_status;
			if ((mtx3354_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
				return mtx3354_status;
			if (FindPattern (tabByte, 0, -1, filename, 0, 0) != -1) 
				{
				if (overwrite) 
					{
					if ((mtx3354_status = viPrintf (instrSession, "MMEM:DEL \"%s\"\r", filename)) < 0)
						return mtx3354_status;
					}
                else return VI_ERROR_MTX3354_EXISTING_FILE;
				}
			/*-Perform transfer----------------------------------------------*/
			if ((mtx3354_status = viPrintf (instrSession, "MMEM:STORE:STATE \"%s\"\r",filename)) < 0)
				return mtx3354_status;
			}
		/*-Target drive is PC------------------------------------------------*/
		else 
			{
			/*-Check overwrite-----------------------------------------------*/    
			if (GetFileInfo(filename,&fileSize)  && !overwrite) 
				return VI_ERROR_MTX3354_EXISTING_FILE;
            
			/*-Modify Protocol for binary file transfers---------------------*/
			viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
			if(interface==VI_INTF_ASRL)
				{
				if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
					return mtx3354_status;
				}
			/*-Perform transfer----------------------------------------------*/    
			if ((mtx3354_status = viPrintf (instrSession,"SYST:SET?\r")) < 0)
				return mtx3354_status;
            
			if (interface==VI_INTF_ASRL) 
				{
				/* Read 11 characters (#an header) to define the length of the binary block following */
				if ((mtx3354_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
				    return mtx3354_status;

				headerSize = tabByte[1] - 0x30;
				Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);
				if ((mtx3354_status = viRead (instrSession, tabByte + HEADER_SIZE, fileSize+4+headerSize-HEADER_SIZE, &retCnt)) < 0)
				     return mtx3354_status;

				retCnt += HEADER_SIZE;

				/*-Reset Protocol for binary file transfers----------------------------*/
				if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
				    return mtx3354_status;
				}
			else if ((mtx3354_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
				return mtx3354_status;
            
			/*-Fill PC file--------------------------------------------------*/    
			if ((ArrayToFile (filename, tabByte, VAL_UNSIGNED_CHAR, retCnt - 2,retCnt - 2, VAL_GROUPS_TOGETHER,
					VAL_GROUPS_AS_ROWS,VAL_CONST_WIDTH, 4, VAL_BINARY,VAL_TRUNCATE)) < 0) 
				return VI_ERROR_INSTR_FILE_WRITE;
			}
		}
    
	/*-Recall setup to MTX3354------------------------------------------------*/
	else 
		{
		/*-Source drive is MTX3354--------------------------------------------*/
		if (drive) 
			{
			/*-Check file name-----------------------------------------------*/    
			if ((mtx3354_status = viPrintf (instrSession,"MMEM:CAT?\r")) < 0)
				return mtx3354_status;
			if ((mtx3354_status = viRead (instrSession, tabByte, MAX_MEM_SIZE,&retCnt)) < 0)
				return mtx3354_status;
			if (FindPattern (tabByte, 0, -1, filename, 0, 0) < 0)
				return VI_ERROR_MTX3354_FILE_NOT_FOUND;
				
			/*-Perform transfer----------------------------------------------*/
			if ((mtx3354_status = viPrintf (instrSession,"MMEM:LOAD:STATE \"%s\"\r",filename)) < 0)
				return mtx3354_status;
			}
			
		/*-Source drive is PC------------------------------------------------*/
		else 
			{
			/*-Check file name-----------------------------------------------*/
			if (!GetFileInfo(filename,&fileSize)) 
				return VI_ERROR_MTX3354_FILE_NOT_FOUND;
			/*-Read PC file--------------------------------------------------*/
			if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, 20, 20,VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
				return VI_ERROR_INSTR_FILE_OPEN;
			Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);
			headerSize = 6;
			if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, headerSize + fileSize, 
					headerSize + fileSize, VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
				return VI_ERROR_INSTR_FILE_OPEN;
			/*-Perform transfer----------------------------------------------*/
			Fmt(string,"%s<SYST:SET %s[t-w*q]\r", headerSize + fileSize, tabByte);
			if ((mtx3354_status = viWrite (instrSession, string, NumFmtdBytes (), &retCnt)) < 0) 
				return mtx3354_status;
			}																  
		Delay(0.5);	
		}
    
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;
	 
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: File Transfer                                                   */ 
/* Purpose:  This function allows you to transfer a file from a drive        */
/*           to another.                                                     */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_fileTransfer (ViSession instrSession, ViString filename,
							ViInt16 srcDrive, ViBoolean overwrite)
{
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte trueFilename[MAX_FILENAME_LEN];
	ViInt16 headerSize;
	ViInt32 retCnt,fileSize, RunState;
	static ViByte tabByte[MAX_MEM_SIZE],tempString[20];
	ViUInt16 interface;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3354_invalidViInt16Range(srcDrive,0,1))
		return VI_ERROR_PARAMETER3;
	if (mtx3354_invalidViBooleanRange(overwrite))
		return VI_ERROR_PARAMETER4;
  
	/*-Check filename--------------------------------------------------------*/
	if ((FindPattern (filename, 0, -1, ".", 0, 0) > 8) && (StringLength(filename) > 12))
		return VI_ERROR_PARAMETER2;

	/*-Computer to MTX3354 transfer-------------------------------------------*/
	if (!srcDrive ) 
		{
		/*-Check file name---------------------------------------------------*/    
		if (!GetFileInfo(filename,&fileSize)) 
			return VI_ERROR_MTX3354_FILE_NOT_FOUND;
		SplitPath (filename, NULL, NULL, trueFilename);

		/* if oscilloscope is in Run mode ?*/
		viWrite (instrSession, "TRIG:RUN:STAT?\r", 15, &RunState);
		viScanf (instrSession, "%hd", &RunState);
		if (RunState)
			viWrite (instrSession, "TRIG:RUN:STAT 0\r", 16, &RunState);

		/*-Check overwrite---------------------------------------------------*/    
		if ((mtx3354_status = viPrintf (instrSession, "MMEM:CAT?\r")) < 0)
			return mtx3354_status;
		if ((mtx3354_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mtx3354_status;
		if (FindPattern (tabByte, 0, -1, filename, 0, 0) != -1) 
			{
			if (overwrite) 
				{
				if ((mtx3354_status = viPrintf (instrSession, "MMEM:DEL \"%s\"\r", filename)) < 0)
					return mtx3354_status;
				}
			else   
				return VI_ERROR_MTX3354_EXISTING_FILE;
			}
    
		/*-Read PC file------------------------------------------------------*/
		GetFileInfo (filename, &fileSize);
     
		/*-Read any file but not ".TRC" or ".CFG"----------------------------*/
		if ((FindPattern(trueFilename, 0, -1, "TRC", 0, 0)==-1) && (FindPattern(trueFilename, 0, -1, "CFG", 0, 0)==-1)) 
			{
			/*-Perform transfer----------------------------------------------*/
			Fmt (tempString, "%i[b4]", fileSize);
			Fmt (tabByte, "MMEM:DATA \"%s\", #%i[b2]%i[b4]", trueFilename, StringLength(tempString), fileSize);
			}
        
		/*-Read ".TRC" or ".CFG" files---------------------------------------*/
		else 
			{
			/*-Perform transfer--------------------------------------------------*/
			Fmt (tabByte, "MMEM:DATA \"%s\",", trueFilename);
			}
			
		headerSize = StringLength(tabByte);

		if (FileToArray (filename, tabByte + headerSize, VAL_UNSIGNED_CHAR, fileSize, 1,
		                	VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
			return VI_ERROR_INSTR_FILE_OPEN;

		tabByte[headerSize + fileSize] = '\r';
		
		if ((mtx3354_status = viWrite (instrSession, tabByte, headerSize + fileSize + 1, &retCnt)) < 0) 
			return mtx3354_status;
		Delay(1.5);	
		
		/*Restart the oscilloscope in Run mode*/
		if (RunState)
			viWrite (instrSession, "TRIG:RUN:STAT 1\r", 16, &RunState);
		}
    
	/*-MTX3354 to Computer transfer-------------------------------------------*/
	else 
		{
		/*-Check filename----------------------------------------------------*/
		if ((mtx3354_status = viPrintf (instrSession, "MMEM:CAT?\r")) < 0)
			return mtx3354_status;
		if ((mtx3354_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mtx3354_status;
		if (FindPattern (tabByte, 0, -1, filename, 0, 0) < 0)
			return VI_ERROR_MTX3354_FILE_NOT_FOUND;

		/*-Check overwrite---------------------------------------------------*/    
		if (GetFileInfo(filename, &fileSize)  && !overwrite) 
			return VI_ERROR_MTX3354_EXISTING_FILE;

		/*-Modify Time Out for big file transfers----------------------------*/
		if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_TRANSFER)) < 0)
			return mtx3354_status;

		/*-Modify Protocol for binary file transfers-------------------------*/
		viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
		if(interface==VI_INTF_ASRL)
			{
			if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
				return mtx3354_status;
			}

		/*-Perform transfer--------------------------------------------------*/
		if ((mtx3354_status = viPrintf (instrSession, "MMEM:DATA? \"%s\"\r", filename)) < 0)
			return mtx3354_status;

		if (interface==VI_INTF_ASRL) 
        	{
			/* Read 11 characters (#an header) to define the length of the binary block following */
			if ((mtx3354_status = viRead (instrSession, tabByte, HEADER_FILE, &retCnt)) < 0)
				return mtx3354_status;

			headerSize = tabByte[1] - 0x30;
			Scan(tabByte, "%s[i2]>%i[b4]", &fileSize);

			if ((mtx3354_status = viRead (instrSession, tabByte+HEADER_FILE, fileSize+4+headerSize-HEADER_FILE, &retCnt)) < 0)
				return mtx3354_status;

			retCnt += HEADER_FILE;

			/*-Reset Protocol for binary file transfers----------------------------*/
			if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
				return mtx3354_status;
			}
		else if ((mtx3354_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mtx3354_status;

		/*-Reset Time Out ---------------------------------------------------*/
		if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
			return mtx3354_status;

		/*-Parse non ".TRC" or ".CFG" files----------------------------------*/
		if ((FindPattern(filename,0,-1,"TRC",0,0)==-1) && (FindPattern(filename, 0, -1, "CFG", 0, 0)==-1))
			headerSize = tabByte[1] - 0x30 + 2; 
		else
			headerSize = 0;

		/*-Fill PC file------------------------------------------------------*/    
		if ((ArrayToFile (filename, tabByte + headerSize, VAL_UNSIGNED_CHAR, retCnt - headerSize - 2, 1, 
		VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_CONST_WIDTH, 4, VAL_BINARY, VAL_TRUNCATE)) < 0) 
			return VI_ERROR_INSTR_FILE_WRITE;
		}

	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3354_status = mtx3354_instrStatus(instrSession)) < 0)
		return mtx3354_status;

	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Write To Instrument                                             */
/* Purpose:  This function writes a command string to the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_writeInstrData (ViSession instrSession, ViString writeBuffer)
{
	ViStatus mtx3354_status = VI_SUCCESS;

	if ((mtx3354_status = viPrintf (instrSession, "%s", writeBuffer)) < 0)
		return mtx3354_status;

	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Read Instrument Buffer                                          */
/* Purpose:  This function reads the output buffer of the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_readInstrData (ViSession instrSession, ViInt32 numBytes,
							ViChar _VI_FAR rdBuf[], ViPInt32 bytesRead)
{
	ViStatus mtx3354_status = VI_SUCCESS;
	*bytesRead = 0L;

	if ((mtx3354_status = viRead (instrSession, rdBuf, BUFFER_SIZE, bytesRead)) < 0)
		return mtx3354_status;

	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Reset                                                           */
/* Purpose:  This function resets the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC  mtx3354_reset (ViSession instrSession)
{
	ViUInt32 retCnt = 0;
	ViStatus  mtx3354_status = VI_SUCCESS;

	/*-Initialize the instrument to a known state----------------------------*/
	if (( mtx3354_status = viWrite (instrSession, "\r*RST\r", 6, &retCnt)) < 0)
		return  mtx3354_status;

	if (( mtx3354_status =  mtx3354_defaultInstrSetup (instrSession)) < 0)  
		return  mtx3354_status;
        
	return  mtx3354_status;
}

/*===========================================================================*/
/* Function: Self-Test                                                       */
/* Purpose:  This function executes the instrument self-test and returns     */
/*           the result.                                                     */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_selfTest (ViSession instrSession, ViPInt16 testResult,
								ViChar _VI_FAR testMessage[])
{
	ViStatus mtx3354_status = VI_SUCCESS;
	ViUInt32 retCnt = 0;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE,20 * TMO_VALUE)) < 0)
		return mtx3354_status; 
    
	if ((mtx3354_status = viWrite (instrSession, "*TST?\r", 6, &retCnt)) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viRead (instrSession, rdBuffer, 50, &retCnt)) < 0)
		return mtx3354_status;
	if (Scan (rdBuffer, "%i[b1]", testResult) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;

	if (*testResult) 
		Fmt(testMessage, "%s<Self test failed");
	else 
		Fmt(testMessage, "%s<No Error");  

	if ((mtx3354_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, 
		TMO_VALUE)) < 0)
		return mtx3354_status;

	return mtx3354_status;
}


/*===========================================================================*/
/* Function: Error Query                                                     */
/* Purpose:  This function queries the instrument error queue, and returns   */
/*           the result. If the error query function is not supported by the */
/*           instrument, this function returns the warning                   */
/*           VI_WARN_NSUP_ERROR_QUERY.                                       */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_errorQuery (ViSession instrSession, ViPInt32 errCode,
							ViChar _VI_FAR errMessage[])
{
	ViUInt32 retCnt = 0;
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mtx3354_status = viWrite (instrSession, "SYST:ERR?\r", 10, &retCnt)) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return mtx3354_status;

	if (Scan (rdBuffer, "%i[b4],%s[t10]", errCode,errMessage) != 2)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;

	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Error Message                                                   */
/* Purpose:  This function translates the error return value from the        */
/*           instrument driver into a user-readable string.                  */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_errorMessage (ViSession instrSession, ViStatus errorCode,
							ViChar _VI_FAR errMessage[])
{
	ViStatus mtx3354_status = VI_SUCCESS;
	ViInt16 i;
    
	static mtx3354_tStringValPair statusDescArray[] = {
		{VI_ERROR_PARAMETER1,   "ERROR: Parameter 1 out of range"},
		{VI_ERROR_PARAMETER2,   "ERROR: Parameter 2 out of range"},
		{VI_ERROR_PARAMETER3,   "ERROR: Parameter 3 out of range"},
		{VI_ERROR_PARAMETER4,   "ERROR: Parameter 4 out of range"},
		{VI_ERROR_PARAMETER5,   "ERROR: Parameter 5 out of range"},
		{VI_ERROR_PARAMETER6,   "ERROR: Parameter 6 out of range"},
		{VI_ERROR_PARAMETER7,   "ERROR: Parameter 7 out of range"},
		{VI_ERROR_PARAMETER8,   "ERROR: Parameter 8 out of range"},
		{VI_ERROR_INSTR_PARAMETER9,   "ERROR: Parameter 9 out of range"},
		{VI_ERROR_FAIL_ID_QUERY,"ERROR: Identification query failed"},
		{VI_ERROR_INSTR_FILE_OPEN,    "ERROR: Opening the specified file"},
		{VI_ERROR_INSTR_FILE_WRITE,   "ERROR: Writing to the specified file"},
		{VI_ERROR_INSTR_INTERPRETING_RESPONSE,           "ERROR: Interpreting the instrument's response"},
		{VI_ERROR_MTX3354_ERROR,                  "ERROR: Instrument specific error"},
		{VI_ERROR_MTX3354_INCONSISTENT_PARAMETER3,"ERROR: Parameter 3 is INCONSISTENT"},
		{VI_ERROR_MTX3354_INCONSISTENT_PARAMETER5,"ERROR: Parameter 5 is INCONSISTENT"},
		{VI_ERROR_MTX3354_INCONSISTENT_PARAMETER6,"ERROR: Parameter 6 is INCONSISTENT"},
		{VI_ERROR_MTX3354_INCONSISTENT_PARAMETER7,"ERROR: Parameter 7 is INCONSISTENT"},
		{VI_ERROR_MTX3354_AUTOTEST_PROBLEM,       "ERROR: Performing a self-test"}, 
		{VI_ERROR_MTX3354_CHAN_NOT_ACTIVE,        "ERROR: Channel is not active"},
		{VI_ERROR_MTX3354_NO_CHAN_ENABLED,        "ERROR: No channel is enabled"},
		{VI_ERROR_MTX3354_FILE_NOT_FOUND,         "ERROR: File not available on support"},
		{VI_ERROR_MTX3354_EXISTING_FILE,          "ERROR: Filename already used"},
		{VI_ERROR_MTX3354_INVALID_FILE_CONTENT,   "ERROR: File content is invalid"},
		{VI_NULL, VI_NULL}  };
	          
	mtx3354_status = viStatusDesc (instrSession, errorCode, errMessage);
	if (mtx3354_status == VI_WARN_UNKNOWN_STATUS) 
		{
		for (i=0; statusDescArray[i].stringName; i++) 
			{
			if (statusDescArray[i].stringVal == errorCode) 
				{
				Fmt (errMessage, "%s<%s", statusDescArray[i].stringName);
				return (VI_SUCCESS);
				}
			}
		Fmt (errMessage, "%s<Unknown Error 0x%x[uw8p0]", errorCode);
		return (VI_WARN_UNKNOWN_STATUS);
		}
	mtx3354_status = VI_SUCCESS;
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Revision Query                                                  */
/* Purpose:  This function returns the driver and instrument revisions.      */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3354_revisionQuery (ViSession instrSession,
					ViChar _VI_FAR driverRev[], ViChar _VI_FAR instrRev[])
{
	ViUInt32 retCnt = 0;
	ViStatus mtx3354_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mtx3354_status = viWrite (instrSession, "*IDN?\r", 6, &retCnt)) < 0)
		return mtx3354_status;
	if ((mtx3354_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return mtx3354_status;
	if (Scan (rdBuffer, "%s[t10]", instrRev) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
	Fmt (driverRev, "%s<%s", MTX3354_REVISION);

	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Close                                                           */
/* Purpose:  This function closes the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC  mtx3354_close (ViSession instrSession)
{

	mtx3354_instrRange instrPtr;
	ViSession rmSession;
	ViStatus  mtx3354_status = VI_SUCCESS;

	if (( mtx3354_status = viGetAttribute (instrSession, VI_ATTR_RM_SESSION, &rmSession)) < 0)
		return  mtx3354_status;
	if (( mtx3354_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr)) < 0)
		return  mtx3354_status;
    
	if(instrPtr != NULL)
		free (instrPtr);
    
	mtx3354_status = viClose (instrSession);
	viClose (rmSession);

	return  mtx3354_status;
}

/*===========================================================================*/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*===========================================================================*/
ViBoolean  mtx3354_invalidViBooleanRange (ViBoolean val)
{
	return ((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Signed Integer Value Out Of Range - ViInt16               */
/* Purpose:  This function checks a short signed integer value to see if it  */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mtx3354_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mtx3354_invalidViInt32Range  (ViInt32 val, ViInt32 min, ViInt32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Unsigned Integer Value Out Of Range - ViUInt16            */
/* Purpose:  This function checks a short unsigned integer value to see if it*/  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mtx3354_invalidViUInt16Range  (ViUInt16 val, ViUInt16 min, ViUInt16 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Unsigned Integer Value Out Of Range - ViUInt32             */
/* Purpose:  This function checks a long unsigned integer value to see if it */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mtx3354_invalidViUInt32Range  (ViUInt32 val, ViUInt32 min, ViUInt32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Float) Value Out Of Range - ViReal32                      */
/* Purpose:  This function checks a real (float) value to see if it lies     */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean  mtx3354_invalidViReal32Range  (ViReal32 val, ViReal32 min, ViReal32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (double) value to see if it lies    */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean  mtx3354_invalidViReal64Range  (ViReal64 val, ViReal64 min, ViReal64 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Instrument Status                                               */
/* Purpose:  This function checks the instrument error status, by reading    */
/*           and processing instrument's event status register. An error     */
/*           is reported if bits 4 and 5 resp. EXE and CME are asserted.     */
/*===========================================================================*/
ViStatus  mtx3354_instrStatus (ViSession instrSession)
{																							 
	ViUInt32 retCnt = 0;
	ViStatus  mtx3354_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViByte status;
    
	/*- Command to query Event Status Register ------------------------------*/
	if (( mtx3354_status = viPrintf (instrSession,"*ESR?\r")) < 0)
		return  mtx3354_status;
        
	if (( mtx3354_status = viRead (instrSession, rdBuffer, 50, &retCnt)) < 0)
		return  mtx3354_status;

	/*- Scanning the response -----------------------------------------------*/
	if (Scan (rdBuffer, "%i[b1]", &status) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    
	/*- Testing bits 4 and 5 ------------------------------------------------*/
	if (status & 0x30) 
		return VI_ERROR_MTX3354_ERROR;
    
	return  mtx3354_status;
}

/*===========================================================================*/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the  mtx3354_init function.  When */
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from  mtx3354_init to     */
/*           VI_NULL.                                                        */
/*===========================================================================*/
ViStatus  mtx3354_initCleanUp (ViSession openRMSession,ViPSession openInstrSession,
                    ViStatus currentStatus)
{
	viClose (*openInstrSession);
	viClose (openRMSession);
	*openInstrSession = VI_NULL;
    
	return currentStatus;
}

/*===========================================================================*/
/* Function: Default Instrument Setup                                        */
/* Purpose:  This function sends a default setup to the instrument.  This    */
/*           function is called by the  mtx3354_reset operation and by the   */
/*            mtx3354_init function if the reset option has not been         */
/*           selected.  This function is useful for configuring any          */
/*           instrument settings that are required by the rest of the        */
/*           instrument driver functions such as turning headers ON or OFF   */
/*           or using the long or short form for commands, queries, and data.*/                                    
/*===========================================================================*/
ViStatus  mtx3354_defaultInstrSetup (ViSession instrSession)
{
	ViStatus  mtx3354_status = VI_SUCCESS;
	ViUInt32 retCnt = 0;
	mtx3354_instrRange instrPtr;
        
	/* Determine if the structure has been initialized for the current VISA  */
	/* Session and malloc if it has not.                                     */
	if ( mtx3354_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
		return  mtx3354_status;
    
	if (instrPtr == NULL) 
		instrPtr = malloc (sizeof (struct  mtx3354_statusDataRanges));
    
	instrPtr -> triggerMode = 0;
	instrPtr -> val2 = 0;
	instrPtr -> val3 = 0;
	Fmt (instrPtr -> instrDriverRevision, "%s<%s",  MTX3354_REVISION);
    
	if (( mtx3354_status = viSetAttribute (instrSession, VI_ATTR_USER_DATA, (ViUInt32)instrPtr)) < 0)
		return  mtx3354_status;                                       

	if (( mtx3354_status = viWrite (instrSession, "*CLS;:FORM INT;:FORM:DINT OFF\r", 30, &retCnt)) < 0)
		return  mtx3354_status;

	return  mtx3354_status;    
}

/*===========================================================================*/
/* Function: Make data readable                                              */
/* Purpose:  This function fills output parameters with readable data        */
/*===========================================================================*/
ViStatus mtx3354_makeDataReadable  (ViByte tabByte[], ViReal64 _VI_FAR waveformArray[], 
						ViPInt32 numberOfSamples, ViPReal64 XStart, ViPReal64 XIncrement)
{
	ViStatus mtx3354_status = VI_SUCCESS;
	ViInt32 i, index, start, type = 0;
	ViReal64 Offset, YIncrement = 0; 
	union 
		{
		ViByte  buffer[4];
		ViInt32 valSample; 
		} Buff;

#define OFFSET_ADC	393216		//0x60000 en hexa

	if ((start = FindPattern (tabByte, 0, 20, "#6200034", 0, 0)) != -1)
		type = 1;
		
	if (type)
		{
		*XIncrement = cmdTimeBase[tabByte[11]] / 50000.0;
		YIncrement = cmdVolt[tabByte[31]] / 32768.0;
		Offset = OFFSET_ADC;
		*numberOfSamples = 50000;
		start += 42;
		}
	else
		{
		/*-Find begining of data file--------------------------------------------*/    
		start = FindPattern (tabByte, 0, 200, "#6200", 0, 0);
		if (start == -1) 
			return VI_ERROR_MTX3354_INVALID_FILE_CONTENT;
		start += 8;
	    index = FindPattern (tabByte, 0, 80, "SCALe", 0, 0);
		Scan(tabByte + index + 6, "%f", XIncrement);

		index = FindPattern (tabByte, 0, 90, "SIZE", 0, 0);
		Scan(tabByte + index + 5, "%i[b4]", numberOfSamples);
		 
		index = FindPattern (tabByte, 0, 200, "OFFSet", 0, 0);
		Scan(tabByte + index + 7, "%f", &Offset);
	
		index = FindPattern (tabByte, 100, 200, "SCALe", 0, 0);
		Scan(tabByte + index + 6, "%f", &YIncrement);
		}
	for (i=0; i<*numberOfSamples; i++)
		{
		Buff.buffer[0] = tabByte[start + i * 4 + 3];
		Buff.buffer[1] = tabByte[start + i * 4 + 2];
		Buff.buffer[2] = tabByte[start + i * 4 + 1];
		Buff.buffer[3] = tabByte[start + i * 4];
		waveformArray[i] = (Buff.valSample - Offset) * YIncrement;
		}
	return mtx3354_status;
}

/*===========================================================================*/
/* Function: Check File Name                                                 */
/* Purpose:  This function checks if the file name is properly edited        */		
/*===========================================================================*/
ViBoolean mtx3354_invalidFilename (ViString filename, ViInt16 drive, ViString type)
{
	ViInt16 startIndex = 0;
    
	if (!drive) 
		startIndex = FindPattern (filename, 0, -1, "\\", 0, 1) + 1;
	if (StringLength(filename + startIndex) > 12) 
		return VI_TRUE;
	startIndex = FindPattern (filename, 0, -1, ".", 0, 1);
	if (CompareStrings (filename, startIndex, type, 0, 0) != 0)  
		return VI_TRUE; 
		
	return VI_FALSE;
}

