#include "mtx162.h"
#include <userint.h>
#include <utility.h>
#include <formatio.h>
#include <visa.h>		  
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/*==================================================================================*/
#define MTX162_REVISION    "Rev 1.0, 03/2010,CVI 7.0"/* Instrument driver revision  */
#define BUFFER_SIZE        512L                      /* File I/O buffer size        */
#define INVALID_SAMPLE     0xFFFF                    /* Sample validity criterion   */
#define MAX_MEM_SIZE       100300                    /* Memory needed for file I/O  */
#define TMO_VALUE          3000                      /* Time out value              */
#define HEADER_SIZE        200                       /* Size of binary data header  */
#define HEADER_FILE        20                        /* Size of binary file header  */
#define USB_VID			   "0x2177"					 /* Vendor ID for USB Device	*/
#define USB_PID			   "0x1002"					 /* Product ID for USB Device	*/
/*==================================================================================*/

/*=MTX162====================================================================*/
/* LabWindows/CVI 7.1.1 Instrument Driver (VISA I/O)                         */
/* Original Release: March 2010                                              */
/*                                                                           */                           
/* Metrix Technical Support: Olivier Lemaire                                 */
/*                                                                           */
/* For LabWindows Technical Support in the United States call:               */
/*              National Instruments, Austin, Texas                          */
/*              Ph (800) 433-3488    Fax (512) 794-5678                      */
/*                                                                           */
/*                                                                           */
/* Modification History: None                                                */
/*                                                                           */
/*===========================================================================*/
 
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS =====================================*/
/*****************************************************************************/
/*-Vertical command arrays---------------------------------------------------*/
static ViString cmdVertCoupling[] = {"AC","DC","GRO"};       
static ViString cmdVoltDiv[] = {"20mV","40mV","80mV", "160mV","400mV","800mV",
						"1.6V","4V","8V", "16V","40V","80V", "160V","400V","800V"}; 
static ViReal64 cmdOffsetMax[] = {80E-3,80E-3,400E-3,400E-3,2.0,2.0,2.5,12.5,12.5,
						64.0,64.0,64.0,640.0,640.0,640.0,6400.0,6400.0,6400.0}; 

/*-Horizontal command arrays-------------------------------------------------*/
static ViString cmdTimeDiv[] = {"1E-9", "2E-9", "5E-9", "10E-9", "25E-9", "50E-9",
								"125E-9", "250E-9", "500E-9", "1E-6", "2.5E-6", "5E-6", "10E-6", "25E-6",
						"50E-6", "100E-6", "200E-6", "500E-6", "1E-3", "2E-3", "5E-3",
						"10E-3", "20E-3", "50E-3", "100E-3", "200E-3", "500E-3","1",
						"2" ,"5", "10", "20", "50", "100", "200"};
                        
static ViReal64 cmdOffset[] = {1E-9, 1E-9, 1E-9, 1E-9, 1E-9, 1E-9, 1E-9, 1E-9, 1E-9,
							1E-6, 1E-6, 1E-6, 1E-6, 1E-6, 1E-6, 1E-6, 1E-6, 1E-6,
							1E-3, 1E-3, 1E-3, 1E-3, 1E-3, 1E-3, 1E-3, 1E-3, 1E-3,
							1,    1,    1,    1,    1,    1,    1,    1};
static ViString cmdcalcfft[] = {"RECT", "HAMM", "HANN", "BLAC"};
static ViString cmdScale[] = {"LIN", "LOG"};

/*-Trigger command arrays---------------------------------------------------*/
static ViString cmdTrigCoupling[] = {"AC","DC","ON"};       
static ViString cmdSlope[] = {"NEG","POS"};
static ViString cmdSource[]= {"INT1", "INT2", "", "", "LINE"}; 

/*-Display command arrays----------------------------------------------------*/
static ViString cmdMaxminAcq[] = {"ENV", "NORM"};
static ViString cmdMode[] = {"NORM", "ENV"};       
static ViString cmdAverage[] = {"0", "2", "4", "8", "16"};       

/*-Read command arrays-------------------------------------------------------*/
static ViString cmdReadMeasure[] = {"MIN", "MAX", "PTP", "LOW", "HIGH", "AMPL", "AC","VOLT",
							"RISE:OVER", "FALL:OVER", "RTIME", "FTIME", "PWID","NWID", "PER", "FREQ", "PDUT",
							"PUL:COUN", "PHAS"};  
static ViReal64 cmdTimeBase[] = {1E-9, 2E-9, 5E-9, 10E-9, 20E-9, 50E-9, 100E-9, 200E-9, 500E-9,
							1E-6, 2E-6, 5E-6, 10E-6, 20E-6, 50E-6, 100E-6, 200E-6, 500E-6, 1E-3,
							2E-3, 5E-3, 10E-3, 20E-3, 50E-3, 100E-3, 200E-3, 500E-3, 1.0, 2.0, 5.0,
							10.0, 20.0, 50.0, 100.0};
static ViReal64 cmdVolt[] = {2.5E-3,5E-3,10E-3,20E-3,50E-3,100E-3,200E-3,500E-3,1.0,
							2.0,5.0,10.0,20.0,50.0,100.0,200.0};
static ViString cmdRef[] = {"INT1", "INT2"};
                                     
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT STATUS/RANGE STRUCTURE  ============================*/
/*****************************************************************************/
/*  mtx162_stringValPair is used in the  mtx162_errorMessage function      */
/*===========================================================================*/
typedef struct   mtx162_stringValPair
{
   ViStatus stringVal;
   ViPString stringName;
}   mtx162_tStringValPair;
	 
struct mtx162_statusDataRanges {
    ViInt16 triggerMode;
    ViInt16 val2;
    ViInt16 val3;
    ViChar instrDriverRevision[256];
    ViInt16 instrMode;
};
typedef struct mtx162_statusDataRanges * mtx162_instrRange;

ViInt16 TrigMode = 0;

/*****************************************************************************/
/*= UTILITY ROUTINE DECLARATIONS (Non-Exportable Functions) =================*/
/*****************************************************************************/
ViBoolean mtx162_invalidViBooleanRange (ViBoolean val);
ViBoolean mtx162_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max);
ViBoolean mtx162_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max);
ViBoolean mtx162_invalidViUInt16Range (ViUInt16 val, ViUInt16 min, ViUInt16 max);
ViBoolean mtx162_invalidViUInt32Range (ViUInt32 val, ViUInt32 min, ViUInt32 max);
ViBoolean mtx162_invalidViReal32Range (ViReal32 val, ViReal32 min, ViReal32 max);
ViBoolean mtx162_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max);
ViStatus  mtx162_makeDataReadable (ViByte tabByte[], ViReal64 _VI_FAR waveformArray[], ViPInt32 numberOfSamples,
                                ViPReal64 XStart, ViPReal64 XIncrement);
ViBoolean mtx162_invalidFilename (ViString filename, ViString type);
ViStatus  mtx162_instrStatus (ViSession instrSession);
ViStatus  mtx162_initCleanUp (ViSession openRMSession, ViPSession openInstrSession, ViStatus currentStatus);
ViStatus  mtx162_defaultInstrSetup (ViSession openInstrSession);

/*****************************************************************************/
/*====== USER-CALLABLE FUNCTIONS (Exportable Functions) =====================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function:	Initialize                                                   */
/* Purpose:		This function opens the instrument, queries the instrument   */
/*				for its ID, and initializes the instrument to a known state  */
/*===========================================================================*/
ViStatus _VI_FUNC  mtx162_init (ViRsrc resourceName, ViBoolean IDQuery,
					ViPSession instrSession)
{
	ViStatus  mtx162_status = VI_SUCCESS;
	ViSession rmSession = 0;
	ViUInt32 retCnt = 0;
	ViByte rdBuffer[BUFFER_SIZE], buff[50];
	ViUInt16 i;
	ViChar InstrumentDescriptor[300];
	ViChar ResourceName[300];
	ViFindList Find_Handle = 0;
    
	/*- Check input parameter ranges ----------------------------------------*/
	if ( mtx162_invalidViBooleanRange (IDQuery))
		return VI_ERROR_PARAMETER2;

	/*- Open instrument session ---------------------------------------------*/
	if (( mtx162_status = viOpenDefaultRM (&rmSession)) < 0)
		return  mtx162_status;

	Fmt (ResourceName, "%s", resourceName);
	if (FindPattern (ResourceName, 0, 3, "USB", 0, 0) != -1)
		{
		Fmt (InstrumentDescriptor, "USB?*RAW{VI_ATTR_MANF_ID == %s && VI_ATTR_MODEL_CODE == %s}", USB_VID, USB_PID);
		if (viFindRsrc (rmSession, InstrumentDescriptor, &Find_Handle, VI_NULL, ResourceName) != 0)
			{
			viClose (rmSession);
			return  mtx162_status;
			}	
		}

	if (( mtx162_status = viOpen (rmSession, ResourceName, VI_NULL, VI_NULL, instrSession)) < 0) 
		{
		viClose (rmSession);
		return  mtx162_status;
		}	

	/*- Configure VISA Formatted I/O ----------------------------------------*/
	if (( mtx162_status = viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
		return  mtx162_initCleanUp (rmSession, instrSession,  mtx162_status);
	if (( mtx162_status = viSetBuf (*instrSession, VI_READ_BUF|VI_WRITE_BUF, 5000)) < 0)
		return  mtx162_initCleanUp (rmSession, instrSession,  mtx162_status);
	if (( mtx162_status = viSetAttribute (*instrSession, VI_ATTR_WR_BUF_OPER_MODE, VI_FLUSH_ON_ACCESS)) < 0)
		return  mtx162_initCleanUp (rmSession, instrSession,  mtx162_status);
	if (( mtx162_status = viSetAttribute (*instrSession, VI_ATTR_RD_BUF_OPER_MODE, VI_FLUSH_ON_ACCESS)) < 0)
		return  mtx162_initCleanUp (rmSession, instrSession,  mtx162_status);
	if (( mtx162_status = viSetAttribute (*instrSession, VI_ATTR_SUPPRESS_END_EN, VI_FALSE)) < 0)
		return  mtx162_initCleanUp (rmSession, instrSession,  mtx162_status);
    mtx162_status = viFlush (*instrSession,VI_WRITE_BUF);
    mtx162_status = viFlush (*instrSession,VI_READ_BUF);

	/*- Identification Query ------------------------------------------------*/
	if (IDQuery) 
		{
		Fmt (buff, "\r*IDN?\r");
		if ((mtx162_status = viWrite (*instrSession, buff, NumFmtdBytes (), &retCnt)) < 0)
			return  mtx162_initCleanUp (rmSession, instrSession,  mtx162_status);
		if ((mtx162_status = viRead (*instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
			return  mtx162_status;
		if (FindPattern (rdBuffer, 0, -1, "MTX162", 1, 0) == -1)
			return  mtx162_initCleanUp (rmSession, instrSession, VI_ERROR_FAIL_ID_QUERY);
		}

	/*- Send Default Instrument Setup ---------------------------------*/
	if (( mtx162_status =  mtx162_defaultInstrSetup (*instrSession)) < 0)
		return  mtx162_initCleanUp (rmSession, instrSession,  mtx162_status);
	
	if (( mtx162_status = viWrite (*instrSession, "FORM INT;FORM:DINT 1\r", 21, &retCnt)) < 0)
		return  mtx162_initCleanUp (rmSession, instrSession,  mtx162_status);
	
	/*-Check instrument status & Return from function------------------------*/
	if (( mtx162_status =  mtx162_instrStatus(*instrSession)) < 0)
		return  mtx162_status;

	return  mtx162_status;
}

/*===========================================================================*/
/* Function: Example                                                         */
/* Purpose:  This function is an example of this driver's functions use.     */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_example (ViSession instrSession, ViInt16 channel,
					ViInt16 timebase, ViInt16 voltageRange, ViInt16 coupling, 
					ViReal64 _VI_FAR data[], ViPInt32 numberOfSamples, 
					ViPReal64 XStart, ViPReal64 XIncrement)
{
	ViStatus mtx162_status = VI_SUCCESS;
    
	/*- Check input parameter ranges ----------------------------------------*/    
	if (mtx162_invalidViInt16Range (channel, 0, 1))
		return VI_ERROR_PARAMETER2;
	if (mtx162_invalidViInt16Range (timebase, 0, 33))
		return VI_ERROR_PARAMETER3;
	if (mtx162_invalidViInt16Range (voltageRange, 0, 17))
		return VI_ERROR_PARAMETER4;
	if (mtx162_invalidViInt16Range (coupling, 0, 2))
		return VI_ERROR_PARAMETER5;
    
	/*- Configuring vertical settings----------------------------------------*/
	if((mtx162_status = mtx162_verticalSetup (instrSession, channel + 1, VI_ON, coupling, 1, voltageRange, 0.0, 0)) < 0)
		return mtx162_status;
        
	/*- Setting timebase ----------------------------------------------------*/
	if((mtx162_status = mtx162_horizontalSetup (instrSession, timebase, -250.0)) < 0)
		return mtx162_status;

	/*- Setting trigger mode to AUTO-----------------------------------------*/
	/*- MTX162 starts acquiring data immediately after it is triggered-------*/
	if((mtx162_status = mtx162_setGeneralTrigger (instrSession, channel, VI_ON, 0, 1, 0.0)) < 0)
		return mtx162_status;
        
	/*- Sending single-shot trigger in wait mode-----------------------------*/
	/*- This function will return as long as the acquisition is running------*/
	if((mtx162_status = mtx162_runAcquisition (instrSession, VI_ON)) < 0)
		return mtx162_status;
    
    Delay (3.0);
    
	/*- Reads trace from the instrument--------------------------------------*/ 
	if((mtx162_status = mtx162_readChToArr (instrSession, channel, data, numberOfSamples, XStart, XIncrement)) < 0)
		return mtx162_status; 
                   
	return mtx162_status;
}
 
/*===========================================================================*/
/* Function: Configure MTX162 Vertical Settings                             */
/* Purpose:  This function configures the vertical settings for a specified  */
/*           channel                                                         */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_verticalSetup (ViSession instrSession, ViInt16 input,
					ViBoolean state, ViInt16 coupling, ViInt16 probe, 
					ViInt16 sense, ViReal64 offset, ViReal64 band)
{
	ViStatus mtx162_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx162_invalidViInt16Range(input, 1, 2))
		return VI_ERROR_PARAMETER2;
	if (mtx162_invalidViBooleanRange(state))
		return VI_ERROR_PARAMETER3;
	if (mtx162_invalidViInt16Range(coupling, 0, 2))
		return VI_ERROR_PARAMETER4;
	if (probe != 1 && probe != 10 && probe != 100 && probe != 1000)
		return VI_ERROR_PARAMETER5;
	if (mtx162_invalidViInt16Range(sense, 0, 17))
		return VI_ERROR_PARAMETER6;
    						  
	switch (probe) 
		{
		/*--Probe x1---------------------------------------------------------*/
		case 1 :	if (mtx162_invalidViInt16Range(sense,0,11))
						return VI_ERROR_MTX162_INCONSISTENT_PARAMETER6;
                    break;
        /*--Probe x10--------------------------------------------------------*/
        case 10 :   if (mtx162_invalidViInt16Range(sense,3,14))
                        return VI_ERROR_MTX162_INCONSISTENT_PARAMETER6; 
					break;
		/*--Probe x100-------------------------------------------------------*/ 
		case 100 :	if (mtx162_invalidViInt16Range(sense,6,17))
						return VI_ERROR_MTX162_INCONSISTENT_PARAMETER6; 
					break;
		/*--Probe x1000------------------------------------------------------*/
		case 1000 :	if (mtx162_invalidViInt16Range(sense,9,20))
						return VI_ERROR_MTX162_INCONSISTENT_PARAMETER6; 
					break;  
		}      
	if (mtx162_invalidViReal64Range(offset,-cmdOffsetMax[sense],cmdOffsetMax[sense]))
		return VI_ERROR_PARAMETER7;
    
	/*-Send vertical config command------------------------------------------*/
	Fmt (wrBuffer, "DISP:TRAC:Y:PDIV%i[b2] 1\r", input);
	if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx162_status; 
	Fmt (wrBuffer, "DISP:TRAC:STAT%i[b2] %i[b2]\r", input, state);
	if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx162_status; 
	Fmt (wrBuffer, "INP%i[b2]:COUP %s\r", input, cmdVertCoupling[coupling]);
	if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx162_status; 
	Fmt (wrBuffer, "BAND%i[b2] %f[e3]\r", input, band);
	if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx162_status; 
	Fmt (wrBuffer, "VOLT%i[b2]:RANG:PTP %s\r", input, cmdVoltDiv[sense]);
	if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx162_status; 
	Fmt (wrBuffer, "VOLT%i[b2]:RANG:OFFS %f\r", input, offset);
	if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx162_status; 
	Fmt (wrBuffer, "DISP:TRAC:Y:PDIV%i[b2] %i[b2]\r", input, probe);
	if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx162_status; 

	/*-Check instrument status & Return from function------------------------*/
	if ((mtx162_status = mtx162_instrStatus(instrSession)) < 0)
		return mtx162_status;
        
	return mtx162_status;
}

/*===========================================================================*/
/* Function: Configure MTX162 Horizontal Settings                           */
/* Purpose:  This function configures the horizontal settings of the         */
/*           instrument                                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_horizontalSetup (ViSession instrSession,
					ViInt16 range, ViReal64 pretrg) 
                  
{
	ViStatus mtx162_status = VI_SUCCESS;
	ViReal64 convBuf;
	ViByte pretrgBuf[BUFFER_SIZE];
	ViInt32 retCntptr;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx162_invalidViInt16Range(range, 0, 34))
		return VI_ERROR_PARAMETER2;
      
	/*----------------------- time bases ------------------------------------*/
	Fmt (pretrgBuf, "DISP:TRAC:X:PDIV %s\r", cmdTimeDiv[range]);
	if ((mtx162_status = viWrite (instrSession, pretrgBuf, StringLength(pretrgBuf), &retCntptr)) < 0)
		return mtx162_status; 
        
	/*-Check input parameter ranges and test pretrig-------------------------*/    
	Fmt(&convBuf, "%f<%s", cmdTimeDiv[range]);
	if (mtx162_invalidViReal64Range(pretrg * cmdOffset[range],( -10 * convBuf), (25 * convBuf)))
		return VI_ERROR_PARAMETER3; 
  
	/*----------------  pretrig --------------------------------------------*/       
	Fmt (pretrgBuf, "SWE:OFFS:TIME %f\r", (pretrg * cmdOffset[range]));
	if ((mtx162_status = viWrite (instrSession, pretrgBuf, StringLength(pretrgBuf), &retCntptr)) < 0)
		return mtx162_status; 

	/*-Check instrument status & Return from function------------------------*/
	if ((mtx162_status = mtx162_instrStatus(instrSession)) < 0)
		return mtx162_status;

	return mtx162_status;
}

/*===========================================================================*/
/* Function: General Trigger                                                 */
/* Purpose:  This function sets general parameters for trigger               */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_setGeneralTrigger (ViSession instrSession, ViInt16 channel, ViBoolean slope,
										ViInt16 coupling, ViInt16 type, ViReal64 level)
{
	ViStatus mtx162_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx162_invalidViInt16Range(channel,0,3))
		return VI_ERROR_PARAMETER2;
	if (mtx162_invalidViBooleanRange(slope))
		return VI_ERROR_PARAMETER3;
	if (mtx162_invalidViInt16Range(coupling,0,5))
		return VI_ERROR_PARAMETER4;
	if (mtx162_invalidViInt16Range(type,0,2))
		return VI_ERROR_PARAMETER5;
	if (channel < 4)
		{
		if (mtx162_invalidViReal64Range(level,-800,800))
			return VI_ERROR_PARAMETER6;
		}
	else
		{ 
		if (mtx162_invalidViReal64Range(level,-800E-3,8E-3))
			return VI_ERROR_PARAMETER6;
    	
		}

	/*-Set slope, coupling, level and filter parameters----------------*/
	Fmt (wrBuffer, "TRIG:SEQ1:SLOP %s\r", cmdSlope[slope]);
	if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx162_status;
	if (channel < 4)
		{
		Fmt (wrBuffer, "TRIG:LEV %f\r", level);
		if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx162_status;

		switch (coupling)
			{
			case 0 :
					Fmt (wrBuffer, "TRIG:COUP %s\r", cmdTrigCoupling[coupling]);
					if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
						return mtx162_status;
					break;
			case 1 :
					Fmt (wrBuffer, "TRIG:COUP %s\r", cmdTrigCoupling[coupling]);
					if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
						return mtx162_status;
					break;	
			case 2 :
					Fmt (wrBuffer, "TRIG:FILT:HPAS 1\r");
					if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
						return mtx162_status;
					break;	
			case 3 :
					Fmt (wrBuffer, "TRIG:FILT:LPAS 1\r");
					if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
						return mtx162_status;
					break;	
			}
		}

	/*-Set trigger type and mode parameters----------------*/
	TrigMode = type;
	if (type == 2)
		{
		if (channel == 4)
			Fmt (wrBuffer, ":INIT:NAME LINE\r");
		else
			Fmt (wrBuffer, ":INIT:NAME EDGE\rTRIG:SOUR %s\r", cmdSource[channel]);
		
		if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx162_status; 
		}
	else
		{
		if (channel == 4)
			Fmt (wrBuffer, ":INIT:CONT:NAME LINE,1\r");
		else
			Fmt (wrBuffer, ":INIT:CONT:NAME EDGE,1\rTRIG:SOUR %s\r", cmdSource[channel]);
		
		if (type == 1)
			Fmt (wrBuffer, "%sTRIG:SEQ1:ATRIG 1\r", wrBuffer);
			
		if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx162_status; 
		}
        
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx162_status = mtx162_instrStatus(instrSession)) < 0)
		return mtx162_status;   
    
	return mtx162_status;
}

/*===========================================================================*/
/* Function: Display Waveform                                                */
/* Purpose:  This function sets waveform display.                            */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_setWaveformDisplay (ViSession instrSession, ViInt16 acquisition,
					ViInt16 average_rate, ViInt16 mode)
{
	ViStatus mtx162_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx162_invalidViInt16Range(acquisition, 0,1))
		return VI_ERROR_PARAMETER2;
	if (mtx162_invalidViInt16Range(average_rate, 0, 4))
		return VI_ERROR_PARAMETER3;
	if (mtx162_invalidViInt16Range(mode, 0, 1))
		return VI_ERROR_PARAMETER4;

	/*-Set waveform display parameters---------------------------------------*/
	Fmt (wrBuffer, "AVER:TYPE %s\r", cmdMaxminAcq[acquisition]);
	if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx162_status;
	Fmt (wrBuffer, "DISP:TRACE:MODE %s\r", cmdMode[mode]);
		if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx162_status;
	Fmt (wrBuffer, "AVER:COUN %s\r", cmdAverage[average_rate]);
	if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx162_status;
	
    /*-Check instrument status & Return from function------------------------*/
	if ((mtx162_status = mtx162_instrStatus(instrSession)) < 0)
		return mtx162_status;
        
	return mtx162_status;
}

/*===========================================================================*/
/* Function: Set Cursors                                                     */
/* Purpose:  This function sets MTX162 cursors.                              */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_setCursors (ViSession instrSession, ViInt16 reference, ViBoolean unattcur,
										ViReal64 t1, ViReal64 t2)
{                              
	ViStatus mtx162_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx162_invalidViInt16Range(reference,0,1))
		return VI_ERROR_PARAMETER2;
	if ((mtx162_invalidViBooleanRange(unattcur))) 
		return VI_ERROR_PARAMETER3;
	if (mtx162_invalidViReal64Range(t1,0,2000.0))
		return VI_ERROR_PARAMETER4;
	if (mtx162_invalidViReal64Range(t2,0,2000.0))
		return VI_ERROR_PARAMETER5;
	 
	/*-Set cursors-----------------------------------------------------------*/    
	Fmt (wrBuffer, "DISP:CURS:REF %s\r", cmdRef[reference]);
	if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx162_status;
    
	if ((mtx162_status = viPrintf (instrSession,"DISP:CURS:TIME1:POS %lf\r",t1)) < 0)
		return mtx162_status;
	Delay (1.7);
    	
	if ((mtx162_status = viPrintf (instrSession,"DISP:CURS:TIME2:POS %lf\r",t2)) < 0)
		return mtx162_status;

	Fmt (wrBuffer, "DISP:CURS:AUTO:STAT %i[b2]\r", unattcur);
	if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx162_status;

	/*-Check instrument status & Return from function------------------------*/
	if ((mtx162_status = mtx162_instrStatus(instrSession)) < 0)
		return mtx162_status;
        
	return mtx162_status;
}

/*===========================================================================*/
/* Function: Run Autoset                                                     */
/* Purpose:  This function performs an automatic setup of the MTX162.       */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_runAutoset (ViSession instrSession)
{                              
	ViStatus mtx162_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
    
	Fmt (wrBuffer, "AUTOS:EXE\r");
	if ((mtx162_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx162_status;

	/*-Check instrument status & Return from function------------------------*/
	if ((mtx162_status = mtx162_instrStatus(instrSession)) < 0)
		return mtx162_status;
        
	return mtx162_status;
}

/*===========================================================================*/
/* Function: Run Acquisition                                                 */
/* Purpose:  This function performs an acquisition session.                  */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_runAcquisition (ViSession instrSession, ViBoolean run)
{                              
	ViStatus mtx162_status = VI_SUCCESS;
	ViInt32 retCnt;
	ViByte rdBuffer[BUFFER_SIZE];
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx162_invalidViBooleanRange(run))
		return VI_ERROR_PARAMETER3;
    
	/*-Get current trigger mode----------------------------------------------*/
	Fmt (rdBuffer, "TRIG:RUN:STAT %i[b2]\r", run);
	if ((mtx162_status = viWrite (instrSession, rdBuffer, StringLength(rdBuffer), &retCnt)) < 0)
		return mtx162_status;
		
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx162_status = mtx162_instrStatus(instrSession)) < 0)
		return mtx162_status;
        
    return mtx162_status;
}

/*===========================================================================*/
/* Function: Read Cursors                                                    */
/* Purpose:  This function reads MTX162 cursors.                             */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_readCursors (ViSession instrSession, ViInt16 reference,
					ViPReal64 t1, ViPReal64 t2, ViPReal64 dt,
					ViPReal64 v1, ViPReal64 v2, ViPReal64 dv)
{                              
	ViStatus mtx162_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViInt16 valtphase;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx162_invalidViInt16Range(reference,0,1))
		return VI_ERROR_PARAMETER2;

	/*-Display cursor menu---------------------------------------------------*/
	Fmt (rdBuffer, "DISP:CURS:REF %s\r", cmdRef[reference]);
	if ((mtx162_status = viWrite (instrSession, rdBuffer, StringLength(rdBuffer), &retCnt)) < 0)
		return mtx162_status;
        
	/*-Set cursor positions--------------------------------------------------*/
	if ((mtx162_status = viPrintf (instrSession,"DISP:CURS:TIME1:POS?\r")) < 0)
		return mtx162_status;
	if ((mtx162_status = viScanf (instrSession,"%lf",t1)) < 0)
		return mtx162_status;
	if ((mtx162_status = viPrintf (instrSession,"DISP:CURS:TIME2:POS?\r")) < 0)
		return mtx162_status;
	if ((mtx162_status = viScanf (instrSession,"%lf",t2)) < 0)
		return mtx162_status;
	if ((mtx162_status = viPrintf (instrSession,"DISP:CURS1:VOLT:POS?\r")) < 0)
		return mtx162_status;
	if ((mtx162_status = viScanf (instrSession,"%lf",v1)) < 0)
		return mtx162_status;
	if ((mtx162_status = viPrintf (instrSession,"DISP:CURS:VOLT2:POS?\r")) < 0)
		return mtx162_status;
	if ((mtx162_status = viScanf (instrSession,"%lf",v2)) < 0)
		return mtx162_status;
		
	/*-Calculate 'delta t' & 'delta v'---------------------------------------*/
	*dt = *t2 - *t1;
	*dv = *v2 - *v1;
        
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx162_status = mtx162_instrStatus(instrSession)) < 0)
		return mtx162_status;
        
	return mtx162_status;
}

/*===========================================================================*/
/* Function: Read Channel To Array                                           */
/* Purpose:  This function reads a trace from the instrument.                */
/*           Data is stored in an array                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_readChToArr (ViSession instrSession, ViInt16 source,			   
					ViReal64 _VI_FAR waveformArray[], ViPInt32 numberOfSamples, 
					ViPReal64 XStart, ViPReal64 XIncrement)
{   
	ViStatus mtx162_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViInt32 retCnt,size, index;
	ViInt16 tstRes;
	static ViByte tabByte[MAX_MEM_SIZE];
	ViByte RunStat;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx162_invalidViInt16Range(source,0,1))
		return VI_ERROR_PARAMETER2;
    
	/*-Check if source is active---------------------------------------------*/
	if ((mtx162_status = viPrintf (instrSession,":DISP:TRAC:STAT%hd?\r", source + 1)) < 0)
		return mtx162_status;
	if ((mtx162_status = viRead (instrSession, tabByte, 5, &retCnt)) < 0)
		return mtx162_status;
	Scan (tabByte,"%i[b2]", &tstRes);
	if (tstRes != 1) 
		return VI_ERROR_MTX162_CHAN_NOT_ACTIVE;
    
	/*-Set to Stop to perform transfer ----------------------------------*/
    viWrite (instrSession, "TRIG:RUN:STAT?\r", 15, &retCnt);
    viScanf (instrSession, "%hd", &RunStat);
    if (RunStat)
	    viWrite (instrSession, "TRIG:RUN:STAT 0\r", 16, &retCnt);

	/*-Set data format and start transfert-----------------------------------*/    
	if ((mtx162_status = viPrintf (instrSession,"TRAC? INT%hd\r", source + 1)) < 0)
		return mtx162_status;
	/*-Modify Protocol for binary file transfers-------------------------*/
	if ((mtx162_status = viSetAttribute (instrSession, VI_ATTR_SUPPRESS_END_EN, VI_TRUE)) < 0)
		return mtx162_status;

	/* Read 11 characters (#an header) to define the length of the binary block following */
	if ((mtx162_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
		return mtx162_status;
	index = FindPattern (tabByte, 0, HEADER_SIZE, "#", 0, 0);
	headerSize = tabByte[index + 1] - 0x30;
	Scan(tabByte + index + 2, "%i[b4]", &size);

	if ((mtx162_status = viRead (instrSession, tabByte + HEADER_SIZE, index + size + 7 + headerSize - HEADER_SIZE, &retCnt)) < 0)
		return mtx162_status;

	/*-Reset Protocol for binary file transfers----------------------------*/
	if ((mtx162_status = viSetAttribute (instrSession, VI_ATTR_SUPPRESS_END_EN, VI_FALSE)) < 0)
		return mtx162_status;

	/*-Set to Run if Stop forced ----------------------------------------*/
    if (RunStat)
	    viWrite (instrSession, "TRIG:RUN:STAT 1\r", 16, &retCnt);

	/*-Make data readable----------------------------------------------------*/
	if ((mtx162_status = mtx162_makeDataReadable(tabByte,waveformArray,numberOfSamples,XStart,XIncrement)) < 0)
		return mtx162_status;
                            
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx162_status = mtx162_instrStatus(instrSession)) < 0)
		return mtx162_status;
        
	return mtx162_status;
}

/*===========================================================================*/
/* Function: Read Channel To File                                            */
/* Purpose:  This function reads a trace from the instrument.                */
/*           Data is stored in a file                                        */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_readChToFile ( ViSession instrSession, ViInt16 channel,
                    ViString filename, ViBoolean overwrite)
{                        
	ViStatus mtx162_status = VI_SUCCESS;
	ViInt32 fileSize, retCnt, RunState;													
	static ViByte tabByte[MAX_MEM_SIZE];
    ViInt16 tstRes, headerSize;
    ViString filenameTemp = "___tmp__.trc";

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx162_invalidViInt16Range(channel,0,1))
		return VI_ERROR_PARAMETER2;								 
	if (mtx162_invalidViBooleanRange(overwrite))
		return VI_ERROR_PARAMETER4;
        
	/*-Check if source is active---------------------------------------------*/
	if ((mtx162_status = viPrintf (instrSession,":DISP:TRAC:STAT%hd?\r", channel + 1)) < 0)
		return mtx162_status;
	if ((mtx162_status = viRead (instrSession, tabByte, 5, &retCnt)) < 0)
		return mtx162_status;
	Scan (tabByte,"%i[b2]", &tstRes);
	if (tstRes != 1) 
		return VI_ERROR_MTX162_CHAN_NOT_ACTIVE;
    
	/*-Check filename--------------------------------------------------------*/
	if (mtx162_invalidFilename (filename,".TRC"))
		return VI_ERROR_PARAMETER4;

	/* if oscilloscope is in Run mode ?*/
	viWrite (instrSession, "TRIG:RUN:STAT?\r", 15, &RunState);
	viScanf (instrSession, "%hd", &RunState);
	if (RunState)
		viWrite (instrSession, "TRIG:RUN:STAT 0\r", 16, &RunState);
	
	/*-Store current trace into a PC file------------------------------------*/
	/*-Check overwrite---------------------------------------------------*/
	if (GetFileInfo(filename,&fileSize)  && !overwrite) 
		return VI_ERROR_MTX162_EXISTING_FILE;

	if (GetFileInfo (filename,&fileSize)  && overwrite) 
		DeleteFile (filename);

	/*-Perform transfer--------------------------------------------------*/
	if ((mtx162_status = viPrintf (instrSession,"MMEM:STOR:TRAC INT%hd,\"%s\"\r",
				channel+1, filenameTemp )) < 0)
		return mtx162_status;

	/*-Temp file transfer------------------------------------------------*/
	/*-Perform transfer--------------------------------------------------*/
	if ((mtx162_status = viPrintf (instrSession, "MMEM:DATA? \"%s\"\r", filenameTemp)) < 0)
		return mtx162_status;

	/* Read 11 characters (#an header) to define the length of the binary block following */
	if ((mtx162_status = viRead (instrSession, tabByte, HEADER_FILE, &retCnt)) < 0)
		return mtx162_status;

	headerSize = tabByte[1] - 0x30;
	Scan(tabByte, "%s[i2]>%i[b4]", &fileSize);

	/*-Modify Protocol for binary file transfers-------------------------*/
	if ((mtx162_status = viSetAttribute (instrSession, VI_ATTR_SUPPRESS_END_EN, VI_TRUE)) < 0)
		return mtx162_status;

	if ((mtx162_status = viRead (instrSession, tabByte+HEADER_FILE, fileSize+4+headerSize-HEADER_FILE, &retCnt)) < 0)
		return mtx162_status;

	retCnt += HEADER_FILE;

	/*-Reset Protocol for binary file transfers----------------------------*/
	if ((mtx162_status = viSetAttribute (instrSession, VI_ATTR_SUPPRESS_END_EN, VI_FALSE)) < 0)
		return mtx162_status;

	/*-Fill PC file------------------------------------------------------*/    
	if ((ArrayToFile (filename, tabByte, VAL_UNSIGNED_CHAR, retCnt - 2, 1, 
	VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_CONST_WIDTH, 4, VAL_BINARY, VAL_TRUNCATE)) < 0) 
		return VI_ERROR_INSTR_FILE_WRITE;

	/*-Delete Temp file--------------------------------------------------*/
	if ((mtx162_status = viPrintf (instrSession,"MMEM:DEL \"%s\"\r", filenameTemp)) < 0)
		return mtx162_status;

	/*Restart the oscilloscope in Run mode*/
	if (RunState)
		viWrite (instrSession, "TRIG:RUN:STAT 1\r", 16, &RunState);
  
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx162_status = mtx162_instrStatus(instrSession)) < 0)
		return mtx162_status;
        
	return mtx162_status;
}

/*===========================================================================*/
/* Function: Read File To Array                                              */
/* Purpose:  This function reads a file from the PC or from the instrument   */
/*           Data is stored in an array                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_readFileToArr (ViSession instrSession, ViString filename, 
					ViReal64 _VI_FAR waveformArray[], ViPInt32 numberOfSamples, 
					ViPReal64 XStart, ViPReal64 XIncrement)
{   
	ViStatus mtx162_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViInt32 retCnt,fileSize, size, index;
	static ViByte tabByte[MAX_MEM_SIZE];
    
	/*-Check filename--------------------------------------------------------*/
	if (mtx162_invalidFilename (filename,".TRC"))
		return VI_ERROR_PARAMETER2;
        
	/*-Read file from PC-----------------------------------------------------*/
	/*-Check filename----------------------------------------------------*/
	if (!GetFileInfo(filename,&fileSize)) 
		return VI_ERROR_MTX162_FILE_NOT_FOUND;
    if ((FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, MAX_MEM_SIZE,MAX_MEM_SIZE,
		VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_BINARY)) < 0)
	return VI_ERROR_INSTR_FILE_OPEN;
		
	/*-Make data readable----------------------------------------------------*/
	if ((mtx162_status = mtx162_makeDataReadable(tabByte,waveformArray,numberOfSamples, XStart, XIncrement)) < 0)
		return mtx162_status;
    
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx162_status = mtx162_instrStatus(instrSession)) < 0)
		return mtx162_status;
        
	return mtx162_status;
}

/*===========================================================================*/
/* Function: Save/Recall Setup From/To Instrument                            */
/* Purpose:  This function allows you to save the current setup of the       */
/*           instrument or to recall a setup stored in a file.               */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_saveSetup (ViSession instrSession, ViBoolean mode,
					ViString filename, ViBoolean overwrite)
{
	ViStatus mtx162_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViUInt32 retCnt, fileSize;
	static ViByte string[2000],tabByte[MAX_MEM_SIZE];

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx162_invalidViBooleanRange(mode))
		return VI_ERROR_PARAMETER2;
	if (mtx162_invalidViBooleanRange(overwrite))
		return VI_ERROR_PARAMETER4;

	/*-Check filename--------------------------------------------------------*/
	if (mtx162_invalidFilename (filename,".CFG"))
		return VI_ERROR_PARAMETER3;

	/*-Save setup from MTX162------------------------------------------------*/
	if (mode) 	
		{
		/*-Target drive is PC------------------------------------------------*/
		/*-Check overwrite-----------------------------------------------*/    
		if (GetFileInfo(filename,&fileSize)  && !overwrite) 
			return VI_ERROR_MTX162_EXISTING_FILE;
        
		/*-Perform transfer----------------------------------------------*/    
		if ((mtx162_status = viPrintf (instrSession,"SYST:SET?\r")) < 0)
			return mtx162_status;
        
		/* Read 11 characters (#an header) to define the length of the binary block following */
		if ((mtx162_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
		    return mtx162_status;

		headerSize = tabByte[1] - 0x30;
		Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);

		/*-Modify Protocol for binary file transfers-------------------------*/
		if ((mtx162_status = viSetAttribute (instrSession, VI_ATTR_SUPPRESS_END_EN, VI_TRUE)) < 0)
			return mtx162_status;

		if ((mtx162_status = viRead (instrSession, tabByte + HEADER_SIZE, fileSize+4+headerSize-HEADER_SIZE, &retCnt)) < 0)
		     return mtx162_status;

		retCnt += HEADER_SIZE;

		/*-Reset Protocol for binary file transfers----------------------------*/
		if ((mtx162_status = viSetAttribute (instrSession, VI_ATTR_SUPPRESS_END_EN, VI_FALSE)) < 0)
			return mtx162_status;

		/*-Fill PC file--------------------------------------------------*/    
		if ((ArrayToFile (filename, tabByte, VAL_UNSIGNED_CHAR, retCnt - 2,retCnt - 2, VAL_GROUPS_TOGETHER,
				VAL_GROUPS_AS_ROWS,VAL_CONST_WIDTH, 4, VAL_BINARY,VAL_TRUNCATE)) < 0) 
			return VI_ERROR_INSTR_FILE_WRITE;
		}
    
	/*-Recall setup to MTX162------------------------------------------------*/
	else 
		{
		/*-Source drive is PC------------------------------------------------*/
		/*-Check file name-----------------------------------------------*/
		if (!GetFileInfo(filename,&fileSize)) 
			return VI_ERROR_MTX162_FILE_NOT_FOUND;
		/*-Read PC file--------------------------------------------------*/
		if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, 20, 20,VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
			return VI_ERROR_INSTR_FILE_OPEN;
		Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);
		headerSize = 6;
		if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, headerSize + fileSize, 
				headerSize + fileSize, VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
			return VI_ERROR_INSTR_FILE_OPEN;
		/*-Perform transfer----------------------------------------------*/
		Fmt(string,"%s<SYST:SET %s[t-w*q]\r", headerSize + fileSize, tabByte);
		if ((mtx162_status = viWrite (instrSession, string, NumFmtdBytes (), &retCnt)) < 0) 
			return mtx162_status;

		Delay(0.5);	
		}
    
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx162_status = mtx162_instrStatus(instrSession)) < 0)
		return mtx162_status;
	 
	return mtx162_status;
}

/*===========================================================================*/
/* Function: Write To Instrument                                             */
/* Purpose:  This function writes a command string to the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_writeInstrData (ViSession instrSession, ViString writeBuffer)
{
	ViStatus mtx162_status = VI_SUCCESS;

	if ((mtx162_status = viPrintf (instrSession, "%s", writeBuffer)) < 0)
		return mtx162_status;

	return mtx162_status;
}

/*===========================================================================*/
/* Function: Read Instrument Buffer                                          */
/* Purpose:  This function reads the output buffer of the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_readInstrData (ViSession instrSession, ViInt32 numBytes,
							ViChar _VI_FAR rdBuf[], ViPInt32 bytesRead)
{
	ViStatus mtx162_status = VI_SUCCESS;
	*bytesRead = 0L;

	if ((mtx162_status = viRead (instrSession, rdBuf, BUFFER_SIZE, bytesRead)) < 0)
		return mtx162_status;

	return mtx162_status;
}

/*===========================================================================*/
/* Function: Reset                                                           */
/* Purpose:  This function resets the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC  mtx162_reset (ViSession instrSession)
{
	ViUInt32 retCnt = 0;
	ViStatus  mtx162_status = VI_SUCCESS;

	/*-Initialize the instrument to a known state----------------------------*/
	if (( mtx162_status = viWrite (instrSession, "\r*RST\r", 6, &retCnt)) < 0)
		return  mtx162_status;

	if (( mtx162_status =  mtx162_defaultInstrSetup (instrSession)) < 0)  
		return  mtx162_status;
        
	return  mtx162_status;
}

/*===========================================================================*/
/* Function: Self-Test                                                       */
/* Purpose:  This function executes the instrument self-test and returns     */
/*           the result.                                                     */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_selfTest (ViSession instrSession, ViPInt16 testResult,
								ViChar _VI_FAR testMessage[])
{
	ViStatus mtx162_status = VI_SUCCESS;
	ViUInt32 retCnt = 0;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mtx162_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE,20 * TMO_VALUE)) < 0)
		return mtx162_status; 
    
	if ((mtx162_status = viWrite (instrSession, "*TST?\r", 6, &retCnt)) < 0)
		return mtx162_status;
	if ((mtx162_status = viRead (instrSession, rdBuffer, 50, &retCnt)) < 0)
		return mtx162_status;
	if (Scan (rdBuffer, "%i[b1]", testResult) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;

	if (*testResult) 
		Fmt(testMessage, "%s<Self test failed");
	else 
		Fmt(testMessage, "%s<No Error");  

	if ((mtx162_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, 
		TMO_VALUE)) < 0)
		return mtx162_status;

	return mtx162_status;
}


/*===========================================================================*/
/* Function: Error Query                                                     */
/* Purpose:  This function queries the instrument error queue, and returns   */
/*           the result. If the error query function is not supported by the */
/*           instrument, this function returns the warning                   */
/*           VI_WARN_NSUP_ERROR_QUERY.                                       */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_errorQuery (ViSession instrSession, ViPInt32 errCode,
							ViChar _VI_FAR errMessage[])
{
	ViUInt32 retCnt = 0;
	ViStatus mtx162_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mtx162_status = viWrite (instrSession, "SYST:ERR?\r", 10, &retCnt)) < 0)
		return mtx162_status;
	if ((mtx162_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return mtx162_status;

	if (Scan (rdBuffer, "%i[b4],%s[t10]", errCode,errMessage) != 2)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;

	return mtx162_status;
}

/*===========================================================================*/
/* Function: Error Message                                                   */
/* Purpose:  This function translates the error return value from the        */
/*           instrument driver into a user-readable string.                  */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_errorMessage (ViSession instrSession, ViStatus errorCode,
							ViChar _VI_FAR errMessage[])
{
	ViStatus mtx162_status = VI_SUCCESS;
	ViInt16 i;
    
	static mtx162_tStringValPair statusDescArray[] = {
		{VI_ERROR_PARAMETER1,   "ERROR: Parameter 1 out of range"},
		{VI_ERROR_PARAMETER2,   "ERROR: Parameter 2 out of range"},
		{VI_ERROR_PARAMETER3,   "ERROR: Parameter 3 out of range"},
		{VI_ERROR_PARAMETER4,   "ERROR: Parameter 4 out of range"},
		{VI_ERROR_PARAMETER5,   "ERROR: Parameter 5 out of range"},
		{VI_ERROR_PARAMETER6,   "ERROR: Parameter 6 out of range"},
		{VI_ERROR_PARAMETER7,   "ERROR: Parameter 7 out of range"},
		{VI_ERROR_PARAMETER8,   "ERROR: Parameter 8 out of range"},
		{VI_ERROR_INSTR_PARAMETER9,   "ERROR: Parameter 9 out of range"},
		{VI_ERROR_FAIL_ID_QUERY,"ERROR: Identification query failed"},
		{VI_ERROR_INSTR_FILE_OPEN,    "ERROR: Opening the specified file"},
		{VI_ERROR_INSTR_FILE_WRITE,   "ERROR: Writing to the specified file"},
		{VI_ERROR_INSTR_INTERPRETING_RESPONSE,           "ERROR: Interpreting the instrument's response"},
		{VI_ERROR_MTX162_ERROR,                  "ERROR: Instrument specific error"},
		{VI_ERROR_MTX162_INCONSISTENT_PARAMETER3,"ERROR: Parameter 3 is INCONSISTENT"},
		{VI_ERROR_MTX162_INCONSISTENT_PARAMETER5,"ERROR: Parameter 5 is INCONSISTENT"},
		{VI_ERROR_MTX162_INCONSISTENT_PARAMETER6,"ERROR: Parameter 6 is INCONSISTENT"},
		{VI_ERROR_MTX162_INCONSISTENT_PARAMETER7,"ERROR: Parameter 7 is INCONSISTENT"},
		{VI_ERROR_MTX162_AUTOTEST_PROBLEM,       "ERROR: Performing a self-test"}, 
		{VI_ERROR_MTX162_CHAN_NOT_ACTIVE,        "ERROR: Channel is not active"},
		{VI_ERROR_MTX162_NO_CHAN_ENABLED,        "ERROR: No channel is enabled"},
		{VI_ERROR_MTX162_FILE_NOT_FOUND,         "ERROR: File not available on support"},
		{VI_ERROR_MTX162_EXISTING_FILE,          "ERROR: Filename already used"},
		{VI_ERROR_MTX162_INVALID_FILE_CONTENT,   "ERROR: File content is invalid"},
		{VI_NULL, VI_NULL}  };
	          
	mtx162_status = viStatusDesc (instrSession, errorCode, errMessage);
	if (mtx162_status == VI_WARN_UNKNOWN_STATUS) 
		{
		for (i=0; statusDescArray[i].stringName; i++) 
			{
			if (statusDescArray[i].stringVal == errorCode) 
				{
				Fmt (errMessage, "%s<%s", statusDescArray[i].stringName);
				return (VI_SUCCESS);
				}
			}
		Fmt (errMessage, "%s<Unknown Error 0x%x[uw8p0]", errorCode);
		return (VI_WARN_UNKNOWN_STATUS);
		}
	mtx162_status = VI_SUCCESS;
	return mtx162_status;
}

/*===========================================================================*/
/* Function: Revision Query                                                  */
/* Purpose:  This function returns the driver and instrument revisions.      */
/*===========================================================================*/
ViStatus _VI_FUNC mtx162_revisionQuery (ViSession instrSession,
					ViChar _VI_FAR driverRev[], ViChar _VI_FAR instrRev[])
{
	ViUInt32 retCnt = 0;
	ViStatus mtx162_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mtx162_status = viWrite (instrSession, "*IDN?\r", 6, &retCnt)) < 0)
		return mtx162_status;
	if ((mtx162_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return mtx162_status;
	if (Scan (rdBuffer, "%s[t10]", instrRev) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
	Fmt (driverRev, "%s<%s", MTX162_REVISION);

	return mtx162_status;
}

/*===========================================================================*/
/* Function: Close                                                           */
/* Purpose:  This function closes the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC  mtx162_close (ViSession instrSession)
{

	mtx162_instrRange instrPtr;
	ViSession rmSession;
	ViStatus  mtx162_status = VI_SUCCESS;

	if (( mtx162_status = viGetAttribute (instrSession, VI_ATTR_RM_SESSION, &rmSession)) < 0)
		return  mtx162_status;
	if (( mtx162_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr)) < 0)
		return  mtx162_status;
    
	if(instrPtr != NULL)
		free (instrPtr);
    
	mtx162_status = viClose (instrSession);
	viClose (rmSession);

	return  mtx162_status;
}

/*===========================================================================*/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*===========================================================================*/
ViBoolean  mtx162_invalidViBooleanRange (ViBoolean val)
{
	return ((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Signed Integer Value Out Of Range - ViInt16               */
/* Purpose:  This function checks a short signed integer value to see if it  */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mtx162_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mtx162_invalidViInt32Range  (ViInt32 val, ViInt32 min, ViInt32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Unsigned Integer Value Out Of Range - ViUInt16            */
/* Purpose:  This function checks a short unsigned integer value to see if it*/  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mtx162_invalidViUInt16Range  (ViUInt16 val, ViUInt16 min, ViUInt16 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Unsigned Integer Value Out Of Range - ViUInt32             */
/* Purpose:  This function checks a long unsigned integer value to see if it */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mtx162_invalidViUInt32Range  (ViUInt32 val, ViUInt32 min, ViUInt32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Float) Value Out Of Range - ViReal32                      */
/* Purpose:  This function checks a real (float) value to see if it lies     */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean  mtx162_invalidViReal32Range  (ViReal32 val, ViReal32 min, ViReal32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (double) value to see if it lies    */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean  mtx162_invalidViReal64Range  (ViReal64 val, ViReal64 min, ViReal64 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Instrument Status                                               */
/* Purpose:  This function checks the instrument error status, by reading    */
/*           and processing instrument's event status register. An error     */
/*           is reported if bits 4 and 5 resp. EXE and CME are asserted.     */
/*===========================================================================*/
ViStatus  mtx162_instrStatus (ViSession instrSession)
{																							 
	ViUInt32 retCnt = 0;
	ViStatus  mtx162_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViByte status;
    
	/*- Command to query Event Status Register ------------------------------*/
	if (( mtx162_status = viPrintf (instrSession,"*ESR?\r")) < 0)
		return  mtx162_status;
        
	if (( mtx162_status = viRead (instrSession, rdBuffer, 50, &retCnt)) < 0)
		return  mtx162_status;

	/*- Scanning the response -----------------------------------------------*/
	if (Scan (rdBuffer, "%i[b1]", &status) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    
	/*- Testing bits 4 and 5 ------------------------------------------------*/
	if (status & 0x30) 
		return VI_ERROR_MTX162_ERROR;
    
	return  mtx162_status;
}

/*===========================================================================*/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the  mtx162_init function.  When */
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from  mtx162_init to     */
/*           VI_NULL.                                                        */
/*===========================================================================*/
ViStatus  mtx162_initCleanUp (ViSession openRMSession,ViPSession openInstrSession,
                    ViStatus currentStatus)
{
	viClose (*openInstrSession);
	viClose (openRMSession);
	*openInstrSession = VI_NULL;
    
	return currentStatus;
}

/*===========================================================================*/
/* Function: Default Instrument Setup                                        */
/* Purpose:  This function sends a default setup to the instrument.  This    */
/*           function is called by the  mtx162_reset operation and by the   */
/*            mtx162_init function if the reset option has not been         */
/*           selected.  This function is useful for configuring any          */
/*           instrument settings that are required by the rest of the        */
/*           instrument driver functions such as turning headers ON or OFF   */
/*           or using the long or short form for commands, queries, and data.*/                                    
/*===========================================================================*/
ViStatus  mtx162_defaultInstrSetup (ViSession instrSession)
{
	ViStatus  mtx162_status = VI_SUCCESS;
	ViUInt32 retCnt = 0;
	mtx162_instrRange instrPtr;
        
	/* Determine if the structure has been initialized for the current VISA  */
	/* Session and malloc if it has not.                                     */
	if ( mtx162_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
		return  mtx162_status;
    
	if (instrPtr == NULL) 
		instrPtr = malloc (sizeof (struct  mtx162_statusDataRanges));
    
	instrPtr -> triggerMode = 0;
	instrPtr -> val2 = 0;
	instrPtr -> val3 = 0;
	Fmt (instrPtr -> instrDriverRevision, "%s<%s",  MTX162_REVISION);
    
	if (( mtx162_status = viSetAttribute (instrSession, VI_ATTR_USER_DATA, (ViUInt32)instrPtr)) < 0)
		return  mtx162_status;                                       

	if (( mtx162_status = viWrite (instrSession, "*CLS;:FORM INT;:FORM:DINT OFF\r", 30, &retCnt)) < 0)
		return  mtx162_status;

	return  mtx162_status;    
}

/*===========================================================================*/
/* Function: Make data readable                                              */
/* Purpose:  This function fills output parameters with readable data        */
/*===========================================================================*/
ViStatus mtx162_makeDataReadable  (ViByte tabByte[], ViReal64 _VI_FAR waveformArray[], 
						ViPInt32 numberOfSamples, ViPReal64 XStart, ViPReal64 XIncrement)
{
	ViStatus mtx162_status = VI_SUCCESS;
	ViInt32 i, index, start, type = 0;
	ViReal64 Offset, YIncrement = 0; 
	union 
		{
		ViByte  buffer[2];
		ViInt16 valSample; 
		} Buff;

//#define OFFSET_ADC	524288		//0x80000 en hexa
#define OFFSET_ADC	2048			//0x0800 en hexa

	if ((start = FindPattern (tabByte, 0, 20, "#6100034", 0, 0)) != -1)
		type = 1;
		
	if (type)
		{
		*XIncrement = cmdTimeBase[tabByte[11]] / 50000.0;
		YIncrement = cmdVolt[tabByte[31]] / 512.0;
		Offset = OFFSET_ADC;
		*numberOfSamples = 50000;
		start += 42;
		}
	else
		{
		/*-Find begining of data file--------------------------------------------*/    
		start = FindPattern (tabByte, 0, 200, "#6100", 0, 0);
		if (start == -1) 
			return VI_ERROR_MTX162_INVALID_FILE_CONTENT;
		start += 8;
	    index = FindPattern (tabByte, 0, 80, "SCALe", 0, 0);
		Scan(tabByte + index + 6, "%f", XIncrement);

		index = FindPattern (tabByte, 0, 90, "SIZE", 0, 0);
		Scan(tabByte + index + 5, "%i[b4]", numberOfSamples);
		 
		index = FindPattern (tabByte, 0, 200, "OFFSet", 0, 0);
		Scan(tabByte + index + 7, "%f", &Offset);
	
		index = FindPattern (tabByte, 100, 200, "SCALe", 0, 0);
		Scan(tabByte + index + 6, "%f", &YIncrement);
		}
		
	for (i=0; i<*numberOfSamples; i++)
		{
		Buff.buffer[0] = tabByte[start + i * 2 + 1];
		Buff.buffer[1] = tabByte[start + i * 2];
		waveformArray[i] = -(OFFSET_ADC - Buff.valSample) * YIncrement;
		}
	return mtx162_status;
}

/*===========================================================================*/
/* Function: Check File Name                                                 */
/* Purpose:  This function checks if the file name is properly edited        */		
/*===========================================================================*/
ViBoolean mtx162_invalidFilename (ViString filename, ViString type)
{
	ViInt16 startIndex = 0;
    
	startIndex = FindPattern (filename, 0, -1, "\\", 0, 1) + 1;
	if (StringLength(filename + startIndex) > 12) 
		return VI_TRUE;
	startIndex = FindPattern (filename, 0, -1, ".", 0, 1);
	if (CompareStrings (filename, startIndex, type, 0, 0) != 0)  
		return VI_TRUE; 
		
	return VI_FALSE;
}

